/*
 * enyx-hfp 通用兼容性头文件
 * 支持从内核3.10开始的所有Linux系统
 * 
 * 支持的系统：
 * - CentOS 7 (内核 3.10.x)
 * - RHEL 7/8/9
 * - Rocky Linux 8/9
 * - Ubuntu 14.04/16.04/18.04/20.04/22.04/24.04
 * - Debian 7/8/9/10/11/12
 * - 其他基于Linux的系统
 * 
 * 内核版本支持：3.10.x - 6.x+
 */

#ifndef ENYX_HFP_UNIVERSAL_COMPATIBILITY_H
#define ENYX_HFP_UNIVERSAL_COMPATIBILITY_H

#include <linux/version.h>

/* 调试信息 */
#ifdef DEBUG
#define HFP_DEBUG(fmt, ...) printk(KERN_DEBUG "HFP: " fmt, ##__VA_ARGS__)
#else
#define HFP_DEBUG(fmt, ...)
#endif

/* 字符串函数兼容性宏定义 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 19, 0)
#    define HFP_STRCPY(dst, src, size) strscpy(dst, src, size)
#else
#    define HFP_STRCPY(dst, src, size) strlcpy(dst, src, size)
#endif

/* DMA 函数兼容性宏定义 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 0, 0)
#    define HFP_DMA_SET_COHERENT_MASK(dev, mask) dma_set_coherent_mask(dev, mask)
#else
#    define HFP_DMA_SET_COHERENT_MASK(dev, mask) pci_set_consistent_dma_mask(to_pci_dev(dev), mask)
#endif

/* 网络设备MAC地址设置兼容性 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 15, 0)
#    define HFP_ETH_ADDR_SET(netdev, addr) eth_hw_addr_set(netdev, addr)
#else
#    define HFP_ETH_ADDR_SET(netdev, addr) memcpy(netdev->dev_addr, addr, netdev->addr_len)
#endif

/* 设备驱动所有者字段兼容性 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 4, 0) && !defined(RHEL_RELEASE_CODE)
#    define HFP_DEVICE_OWNER_FIELD
#elif defined(RHEL_RELEASE_CODE) && LINUX_VERSION_CODE >= KERNEL_VERSION(5, 10, 0)
#    define HFP_DEVICE_OWNER_FIELD
#else
#    define HFP_DEVICE_OWNER_FIELD .owner = THIS_MODULE,
#endif

/* struct class owner字段兼容性 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 4, 0) && !defined(RHEL_RELEASE_CODE)
#    define HFP_CLASS_OWNER_FIELD
#elif defined(RHEL_RELEASE_CODE) && LINUX_VERSION_CODE >= KERNEL_VERSION(5, 10, 0)
#    define HFP_CLASS_OWNER_FIELD
#else
#    define HFP_CLASS_OWNER_FIELD .owner = THIS_MODULE,
#endif

/* 内存映射标志设置兼容性 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 3, 0) || (defined(RHEL_RELEASE_CODE) && LINUX_VERSION_CODE >= KERNEL_VERSION(5, 14, 0))
#    define HFP_VM_FLAGS_SET(vma, flags) vm_flags_set(vma, flags)
#else
#    define HFP_VM_FLAGS_SET(vma, flags) do { (vma)->vm_flags |= (flags); } while(0)
#endif

/* 总线匹配函数兼容性 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 10, 0)
#    define HFP_BUS_MATCH_SIGNATURE static int bus_match(struct device *dev, const struct device_driver *driver)
#else
#    define HFP_BUS_MATCH_SIGNATURE static int bus_match(struct device *dev, struct device_driver *driver)
#endif

/* 总线事件函数兼容性 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(6, 3, 0)
#    define HFP_BUS_UEVENT_SIGNATURE static int bus_uevent(const struct device *dev, struct kobj_uevent_env *env)
#elif defined(RHEL_RELEASE_CODE) && LINUX_VERSION_CODE >= KERNEL_VERSION(5, 14, 0)
#    define HFP_BUS_UEVENT_SIGNATURE static int bus_uevent(const struct device *dev, struct kobj_uevent_env *env)
#else
#    define HFP_BUS_UEVENT_SIGNATURE static int bus_uevent(struct device *dev, struct kobj_uevent_env *env)
#endif

/* MSI中断分配兼容性 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 12, 0)
#    define HFP_ENABLE_MSI(pdev, max_count) pci_enable_msi_range(pdev, 1, max_count)
#else
#    define HFP_ENABLE_MSI(pdev, max_count) pci_enable_msi_block(pdev, max_count)
#endif

/* 网络设备MTU设置兼容性 - 简化版本 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 10, 0)
#    define HFP_SET_MTU_LIMITS(netdev, min_val, max_val) \
        do { \
            netdev->min_mtu = (min_val); \
            netdev->max_mtu = (max_val); \
        } while(0)
#else
#    define HFP_SET_MTU_LIMITS(netdev, min_val, max_val)
#endif

/* NAPI兼容性 - 全局版本 */
#ifndef NAPI_POLL_WEIGHT
#    define NAPI_POLL_WEIGHT 64
#endif

/* 
 * NAPI兼容性分析：
 * - RHEL7 (3.10): netif_napi_add(dev, napi, poll, weight) - 4个参数
 * - RHEL8 (4.18): netif_napi_add(dev, napi, poll) - 3个参数，内部调用netif_napi_add_weight
 * - RHEL9 (5.14): netif_napi_add(dev, napi, poll) - 3个参数，内部调用netif_napi_add_weight
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 19, 0)
#    define HFP_NETIF_NAPI_ADD(netdev, napi, poll_func, weight) netif_napi_add_weight(netdev, napi, poll_func, weight)
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(4, 10, 0)
#    define HFP_NETIF_NAPI_ADD(netdev, napi, poll_func, weight) netif_napi_add(netdev, napi, poll_func)
#else
#    define HFP_NETIF_NAPI_ADD(netdev, napi, poll_func, weight) netif_napi_add(netdev, napi, poll_func, weight)
#endif

/* netdev_priv兼容性 */
#define HFP_NETDEV_PRIV(netdev) netdev_priv(netdev)

/* Fallthrough兼容性 - 全局版本 */
#if __GNUC__ >= 7
#    define HFP_FALLTHROUGH fallthrough;
#elif defined(__clang__) && __clang_major__ >= 10
#    define HFP_FALLTHROUGH [[clang::fallthrough]];
#else
#    define HFP_FALLTHROUGH /* fallthrough */
#endif

/* 版本信息宏 */
#define HFP_COMPATIBILITY_VERSION "rhel-1.0.0"
#define HFP_SUPPORTED_KERNEL_MIN "3.10.0"
#define HFP_SUPPORTED_KERNEL_MAX "6.x+"

/* 兼容性检查宏 */
#define HFP_CHECK_KERNEL_VERSION() \
    do { \
        if (LINUX_VERSION_CODE < KERNEL_VERSION(3, 10, 0)) { \
            printk(KERN_ERR "HFP: Unsupported kernel version. Minimum required: 3.10.0\n"); \
            return -ENOTSUPP; \
        } \
        HFP_DEBUG("HFP: Kernel version check passed. Version: %d.%d.%d\n", \
                  LINUX_VERSION_CODE >> 16, \
                  (LINUX_VERSION_CODE >> 8) & 0xFF, \
                  LINUX_VERSION_CODE & 0xFF); \
    } while(0)

/* 系统信息宏 */
#define HFP_GET_SYSTEM_INFO() \
    do { \
        HFP_DEBUG("HFP: System compatibility information:\n"); \
        HFP_DEBUG("  Kernel version: %d.%d.%d\n", \
                  LINUX_VERSION_CODE >> 16, \
                  (LINUX_VERSION_CODE >> 8) & 0xFF, \
                  LINUX_VERSION_CODE & 0xFF); \
        HFP_DEBUG("  Architecture: %s\n", UTS_MACHINE); \
        HFP_DEBUG("  Compatibility version: %s\n", HFP_COMPATIBILITY_VERSION); \
    } while(0)

#endif /* ENYX_HFP_UNIVERSAL_COMPATIBILITY_H */
