/*
 * HFP tx implementation.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "enyx_hfp_tx_usr.h"

#include <linux/version.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/dma-mapping.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/interrupt.h>

#include "enyx_hfp_common.h"
#include "enyx_hfp_universal_compatibility.h"
#include "enyx_hfp_tx_dev.h"

static DECLARE_BITMAP(device_used, 128);

struct enyx_hfp_tx_data {
    struct enyx_hfp_device * parent;
    struct enyx_hfp_tx_dev * tx_dev;
    struct enyx_hfp_char_device_region * region;
    size_t coherency_line_size;
};

struct enyx_hfp_tx_char_dev {
    atomic_t use_count;
    struct semaphore lock;

    struct enyx_hfp_tx_data * data;
    uint16_t channel_id;

    struct cdev cdev;
    dev_t dev;

    struct device device;
};

#define to_char_dev(n) \
    container_of(n, struct enyx_hfp_tx_char_dev, device)

struct enyx_hfp_tx_char_devs {
    struct enyx_hfp_tx_char_dev * device_ctrl;
    struct enyx_hfp_tx_char_dev * device_buffer;
};

/**
 *
 */
struct enyx_hfp_tx_usr {
    unsigned int index;

    struct enyx_hfp_tx_data data;

    uint16_t char_devs_count;
    struct enyx_hfp_tx_char_devs * char_devs;

    struct device device;
};

#define to_enyx_hfp_tx_usr(n) \
    container_of(n, struct enyx_hfp_tx_usr, device)

static struct class tx_class = {
    .name = "enyx_hfp_tx",
    HFP_CLASS_OWNER_FIELD
};

int
enyx_hfp_tx_class_register(void)
{
    int err;

    err = class_register(&tx_class);
    if (err)
        goto class_register_failed;

    return 0;

class_register_failed:
    return err;
}
EXPORT_SYMBOL(enyx_hfp_tx_class_register);

void
enyx_hfp_tx_class_unregister(void)
{
    class_unregister(&tx_class);
}
EXPORT_SYMBOL(enyx_hfp_tx_class_unregister);

static int
char_dev_ctrl_open(struct inode *inode, struct file *filp)
{
    struct enyx_hfp_tx_char_dev * dev = container_of(inode->i_cdev,
                                                struct enyx_hfp_tx_char_dev,
                                                cdev);
    struct enyx_hfp_tx_dev * tx_dev = dev->data->tx_dev;
    int err;

    if (atomic_cmpxchg(&dev->use_count, 0, 1) != 0) {
        dev_err(&dev->device, "Already open as exclusive\n");
        return -EBUSY;
    }

    if (! enyx_hfp_device_get(dev->data->parent)) {
        err = -ENODEV;
        goto enyx_hfp_device_get_failed;
    }

    err = tx_dev->enable_channel(tx_dev, dev->channel_id);
    if (err < 0)
        goto enyx_hfp_tx_dev_enable_channel_failed;

    filp->private_data = dev;

    return 0;

enyx_hfp_tx_dev_enable_channel_failed:
    enyx_hfp_device_put(dev->data->parent);
enyx_hfp_device_get_failed:
    atomic_dec(&dev->use_count);
    return err;
}

static int
char_dev_ctrl_release(struct inode *inode, struct file *filp)
{
    struct enyx_hfp_tx_char_dev * dev = filp->private_data;
    struct enyx_hfp_tx_dev * tx_dev = dev->data->tx_dev;

    tx_dev->disable_channel(tx_dev, dev->channel_id);

    enyx_hfp_device_put(dev->data->parent);

    atomic_dec(&dev->use_count);

    return 0;
}

static loff_t
char_dev_ctrl_llseek(struct file *filp, loff_t off, int whence)
{
    struct enyx_hfp_tx_char_dev * dev = filp->private_data;
    struct enyx_hfp_tx_dev * tx_dev = dev->data->tx_dev;
    loff_t size = tx_dev->get_ctrl_size(tx_dev) /
                  tx_dev->get_channel_count(tx_dev);
    loff_t newpos;

    if (down_interruptible(&dev->lock))
        return -ERESTARTSYS;

    switch(whence) {
    case 0: /* SEEK_SET */
        newpos = off;
        break;

    case 1: /* SEEK_CUR */
        newpos = filp->f_pos + off;
        break;

    case 2: /* SEEK_END */
        newpos = size;
        break;

    default:
        newpos = -EINVAL;
        break;
    }

    if (newpos < 0 || newpos > size)
        newpos = -EINVAL;

    if (newpos >= 0)
        filp->f_pos = newpos;

    up(&dev->lock);

    return newpos;
}

static int
char_dev_ctrl_mmap(struct file * filp, struct vm_area_struct * vma)
{
    struct enyx_hfp_tx_char_dev * dev = filp->private_data;
    struct enyx_hfp_tx_dev * tx_dev = dev->data->tx_dev;
    int err;

    if (down_interruptible(&dev->lock)) {
        err = -ERESTARTSYS;
        goto lock_failed;
    }

    err = tx_dev->mmap_ctrl(tx_dev, vma, dev->channel_id);

    up(&dev->lock);
lock_failed:
    return err;
}

static const struct file_operations char_dev_ctrl_file_ops = {
    .owner = THIS_MODULE,
    .open = char_dev_ctrl_open,
    .release = char_dev_ctrl_release,
    .llseek = char_dev_ctrl_llseek,
    .mmap = char_dev_ctrl_mmap,
};

static int
char_dev_buffer_open(struct inode *inode, struct file *filp)
{
    struct enyx_hfp_tx_char_dev * dev = container_of(inode->i_cdev,
                                                struct enyx_hfp_tx_char_dev,
                                                cdev);
    int err;

    if (atomic_cmpxchg(&dev->use_count, 0, 1) != 0) {
        dev_err(&dev->device, "Already open as exclusive\n");
        return -EBUSY;
    }

    if (! enyx_hfp_device_get(dev->data->parent)) {
        err = -ENODEV;
        goto enyx_hfp_device_get_failed;
    }

    filp->private_data = dev;

    return 0;

enyx_hfp_device_get_failed:
    atomic_dec(&dev->use_count);
    return err;
}

static int
char_dev_buffer_release(struct inode *inode, struct file *filp)
{
    struct enyx_hfp_tx_char_dev * dev = filp->private_data;

    enyx_hfp_device_put(dev->data->parent);

    atomic_dec(&dev->use_count);

    return 0;
}

static loff_t
char_dev_buffer_llseek(struct file *filp, loff_t off, int whence)
{
    struct enyx_hfp_tx_char_dev * dev = filp->private_data;
    struct enyx_hfp_tx_dev * tx_dev = dev->data->tx_dev;
    loff_t size = tx_dev->get_buffer_size(tx_dev, dev->channel_id);
    loff_t newpos;

    if (down_interruptible(&dev->lock))
        return -ERESTARTSYS;

    switch(whence) {
    case 0: /* SEEK_SET */
        newpos = off;
        break;

    case 1: /* SEEK_CUR */
        newpos = filp->f_pos + off;
        break;

    case 2: /* SEEK_END */
        newpos = size;
        break;

    default:
        newpos = -EINVAL;
        break;
    }

    if (newpos < 0 || newpos > size)
        newpos = -EINVAL;

    if (newpos >= 0)
        filp->f_pos = newpos;

    up(&dev->lock);

    return newpos;
}

static int
char_dev_buffer_mmap(struct file * filp, struct vm_area_struct * vma)
{
    struct enyx_hfp_tx_char_dev * dev = filp->private_data;
    struct enyx_hfp_tx_dev * tx_dev = dev->data->tx_dev;
    int err;

    if (down_interruptible(&dev->lock)) {
        err = -ERESTARTSYS;
        goto lock_failed;
    }

    err = tx_dev->mmap_buffer(tx_dev, vma, dev->channel_id);

    up(&dev->lock);
lock_failed:
    return err;
}

static const struct file_operations char_dev_buffer_file_ops = {
    .owner = THIS_MODULE,
    .open = char_dev_buffer_open,
    .release = char_dev_buffer_release,
    .llseek = char_dev_buffer_llseek,
    .mmap = char_dev_buffer_mmap,
};

static ssize_t
channel_id_show(struct device * device,
                struct device_attribute * attr,
                char * buf)
{
    struct enyx_hfp_tx_char_dev * dev = to_char_dev(device);
    return scnprintf(buf, PAGE_SIZE, "%hu\n", dev->channel_id);
}

static struct device_attribute char_dev_attr_channel_id = __ATTR_RO(channel_id);

static ssize_t
mtu_show(struct device * device,
         struct device_attribute * attr,
         char * buf)
{
    struct enyx_hfp_tx_char_dev * dev = to_char_dev(device);
    struct enyx_hfp_tx_dev * tx_dev = dev->data->tx_dev;
    return scnprintf(buf, PAGE_SIZE, "%zu\n",
                     tx_dev->get_mtu(tx_dev, dev->channel_id));
}

static struct device_attribute char_dev_attr_mtu = __ATTR_RO(mtu);

static ssize_t
buffer_size_show(struct device * device,
         struct device_attribute * attr,
         char * buf)
{
    struct enyx_hfp_tx_char_dev * dev = to_char_dev(device);
    struct enyx_hfp_tx_dev * tx_dev = dev->data->tx_dev;
    return scnprintf(buf, PAGE_SIZE, "%zu\n",
                     tx_dev->get_buffer_size(tx_dev, dev->channel_id));
}

static struct device_attribute char_dev_attr_buffer_size = __ATTR_RO(buffer_size);

static ssize_t
alignment_show(struct device * device,
               struct device_attribute * attr,
               char * buf)
{
    struct enyx_hfp_tx_char_dev * dev = to_char_dev(device);
    return scnprintf(buf, PAGE_SIZE, "%zu\n",
                     dev->data->coherency_line_size);
}

static struct device_attribute char_dev_attr_alignment = __ATTR_RO(alignment);

static ssize_t
channel_usage_show(struct device * device,
                   struct device_attribute * attr,
                   char * buf)
{
    struct enyx_hfp_tx_char_dev * dev = to_char_dev(device);
    struct enyx_hfp_tx_dev * tx_dev = dev->data->tx_dev;
    char usage[32];

    memset(usage, 0, sizeof(usage));
    if (tx_dev->get_usage)
        tx_dev->get_usage(tx_dev, dev->channel_id, usage, sizeof(usage));

    return scnprintf(buf, PAGE_SIZE, "%s\n", usage);
}

static struct device_attribute char_dev_attr_channel_usage
        = __ATTR_RO(channel_usage);

static struct attribute * char_dev_attrs[] = {
    &char_dev_attr_channel_id.attr,
    &char_dev_attr_mtu.attr,
    &char_dev_attr_buffer_size.attr,
    &char_dev_attr_alignment.attr,
    &char_dev_attr_channel_usage.attr,
    NULL,
};

static const struct attribute_group char_dev_group = {
    .attrs = char_dev_attrs,
};

static const struct attribute_group * char_dev_groups[] = {
    &char_dev_group,
    NULL
};

static ssize_t
index_show(struct device * device,
           struct device_attribute * attr,
           char * buf)
{
    struct enyx_hfp_tx_usr * dev = to_enyx_hfp_tx_usr(device);
    return scnprintf(buf, PAGE_SIZE, "%u\n", dev->index);
}

static struct device_attribute tx_attr_index = __ATTR_RO(index);

static ssize_t
channel_count_show(struct device * device,
                   struct device_attribute * attr,
                   char * buf)
{
    struct enyx_hfp_tx_usr * dev = to_enyx_hfp_tx_usr(device);
    struct enyx_hfp_tx_dev * tx_dev = dev->data.tx_dev;
    return scnprintf(buf, PAGE_SIZE, "%zu\n",
                     tx_dev->get_channel_count(tx_dev));
}

static struct device_attribute tx_attr_channel_count = __ATTR_RO(channel_count);

static struct attribute * tx_attrs[] = {
    &tx_attr_index.attr,
    &tx_attr_channel_count.attr,
    NULL,
};

static const struct attribute_group tx_group = {
    .attrs = tx_attrs,
};

static const struct attribute_group * tx_groups[] = {
    &tx_group,
    NULL
};

static void
destroy_char_dev(struct enyx_hfp_tx_char_dev * dev)
{
    device_unregister(&dev->device);
}

static void
release_char_dev(struct device * device)
{
    struct enyx_hfp_tx_char_dev * dev = to_char_dev(device);

    dev_dbg(&dev->device, "Destroying\n");

    cdev_del(&dev->cdev);
    enyx_hfp_char_device_put(dev->data->region, dev->cdev.dev);
    kfree(dev);
}

void
destroy_enyx_hfp_tx_usr(struct enyx_hfp_tx_usr * dev)
{
    while (dev->char_devs_count--) {
        struct enyx_hfp_tx_char_devs * devs = &dev->char_devs[dev->char_devs_count];
        destroy_char_dev(devs->device_buffer);
        destroy_char_dev(devs->device_ctrl);
    }
    kfree(dev->char_devs);

    device_unregister(&dev->device);
}
EXPORT_SYMBOL(destroy_enyx_hfp_tx_usr);

static void
release_tx_device(struct device * device)
{
    struct enyx_hfp_tx_usr * dev = to_enyx_hfp_tx_usr(device);

    dev_dbg(&dev->device, "Destroying\n");

    enyx_hfp_tx_dev_destroy(dev->data.tx_dev);

    clear_bit(dev->index, device_used);
    kfree(dev);
}

static const struct device_type char_dev_type = {
    .name = "enyx_hfp_tx_char",
    .groups = char_dev_groups,
    .release = release_char_dev,
};

static const struct device_type tx_type = {
    .name = "enyx_hfp_tx",
    .groups = tx_groups,
    .release = release_tx_device,
};

static struct enyx_hfp_tx_char_dev *
init_char_dev(struct enyx_hfp_tx_usr * parent,
              const char * name_suffix,
              uint16_t channel_id,
              const struct file_operations * file_ops)
{
    struct enyx_hfp_tx_char_dev * dev = NULL;
    char name[256];
    dev_t dev_id;

    if (snprintf(name, sizeof(name), "enyx_hfp_tx%d_%s%d",
                 parent->index, name_suffix, channel_id) >= sizeof(name)) {
        dev_err(&parent->device, "Can't generate tx device name\n");
        goto snprintf_failed;
    }

    dev = kzalloc(sizeof(*dev), GFP_KERNEL);
    if (! dev) {
        dev_err(&parent->device, "Can't allocate %s\n", name);
        goto kzalloc_failed;
    }

    dev->data = &parent->data;
    dev->channel_id = channel_id;
    dev_id = enyx_hfp_char_device_get(dev->data->region);

    sema_init(&dev->lock, 1);

    cdev_init(&dev->cdev, file_ops);
    dev->cdev.owner = THIS_MODULE;
    if (cdev_add(&dev->cdev, dev_id, 1) < 0) {
        dev_err(&parent->device, "Can't add the %s device %u:%u\n",
                name, MAJOR(dev_id), MINOR(dev_id));
        goto cdev_add_failed;
    }

    dev->device.devt = dev->cdev.dev;
    dev->device.parent = &parent->device;
    dev->device.type = &char_dev_type;
    dev->device.class = &tx_class;
    if (dev_set_name(&dev->device, name) < 0) {
        dev_err(&parent->device, "Can't set %s device name\n", name);
        goto dev_set_name_failed;
    }

    if (device_register(&dev->device) < 0) {
        dev_err(&parent->device, "Can't register %s device\n", name);
        goto device_register_failed;
    }

    dev_dbg(&dev->device, "Created\n");

    return dev;

device_register_failed:
    put_device(&dev->device);
    /* The device will be garbage collected */
    return NULL;
dev_set_name_failed:
    cdev_del(&dev->cdev);
cdev_add_failed:
    enyx_hfp_char_device_put(dev->data->region, dev_id);
    kfree(dev);
kzalloc_failed:
snprintf_failed:
    return NULL;
}

static int
init_char_devs(struct enyx_hfp_tx_usr * dev)
{
    struct enyx_hfp_tx_dev * tx_dev = dev->data.tx_dev;
    uint16_t i;

    dev->char_devs_count = tx_dev->get_channel_count(tx_dev);
    dev->char_devs = kcalloc(dev->char_devs_count, sizeof(*dev->char_devs),
                             GFP_KERNEL);
    if (! dev->char_devs) {
        dev_err(&dev->device, "Can't allocate tx char devices\n");
        goto calloc_failed;
    }

    for (i = 0; i != dev->char_devs_count; ++i) {
        struct enyx_hfp_tx_char_devs * devs = &dev->char_devs[i];

        devs->device_ctrl = init_char_dev(dev, "ctrl", i,
                                         &char_dev_ctrl_file_ops);
        if (! devs->device_ctrl)
            goto init_char_dev_ctrl_failed;

        devs->device_buffer = init_char_dev(dev, "buffer", i,
                                           &char_dev_buffer_file_ops);
        if (! devs->device_buffer)
            goto init_char_dev_buffer_failed;
    }

    return 0;

init_char_dev_buffer_failed:
    destroy_char_dev(dev->char_devs[i].device_ctrl);
init_char_dev_ctrl_failed:
    while (i--) {
        destroy_char_dev(dev->char_devs[i].device_ctrl);
        destroy_char_dev(dev->char_devs[i].device_buffer);
    }
    kfree(dev->char_devs);
calloc_failed:
    return -ENOMEM;
}

struct enyx_hfp_tx_usr *
create_enyx_hfp_tx_usr(struct enyx_hfp_device * parent,
                  struct enyx_hfp_char_device_region * region)
{
    struct enyx_hfp_tx_usr * dev;

    dev = kzalloc(sizeof(*dev), GFP_KERNEL);
    if (! dev) {
        dev_err(&parent->device, "Can't allocate tx\n");
        goto kzalloc_failed;
    }

    dev->data.parent = parent;
    dev->data.coherency_line_size = cache_line_size();
    dev->data.tx_dev = enyx_hfp_tx_dev_create(parent,
                                         dev->data.coherency_line_size);
    if (!dev->data.tx_dev)
        goto enyx_hfp_tx_dev_create_failed;


    dev->index = find_first_zero_bit(device_used, sizeof(device_used));
    set_bit(dev->index, device_used);

    dev->device.parent = &parent->device;
    dev->device.type = &tx_type;
    dev->device.class = &tx_class;
    if (dev_set_name(&dev->device, "enyx_hfp_tx%d", dev->index) < 0) {
        dev_err(&parent->device,
                "Can't set enyx_hfp_tx%d device name\n", dev->index);
        goto dev_set_name_failed;
    }

    if (device_register(&dev->device) < 0) {
        dev_err(&parent->device,
                "Can't register enyx_hfp_tx%d device\n", dev->index);
        goto device_register_failed;
    }

    dev_info(&dev->device, "Created\n");

    dev->data.region = region;

    if (init_char_devs(dev) < 0)
        goto init_char_devs_failed;

    return dev;

init_char_devs_failed:
device_register_failed:
    put_device(&dev->device);
    /* The device will be garbage collected */
    return NULL;
dev_set_name_failed:
    clear_bit(dev->index, device_used);
    enyx_hfp_tx_dev_destroy(dev->data.tx_dev);
enyx_hfp_tx_dev_create_failed:
    kfree(dev);
kzalloc_failed:
    return NULL;
}
EXPORT_SYMBOL(create_enyx_hfp_tx_usr);

