/*
 * HFP mm implementation.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "enyx_hfp_mm_usr.h"

#include <linux/version.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/module.h>

#include "enyx_hfp_common.h"
#include "enyx_hfp_universal_compatibility.h"

static DECLARE_BITMAP(device_used, 128);

static struct class mm_class = {
    .name = "enyx_hfp_mm",
    HFP_CLASS_OWNER_FIELD
};

int
enyx_hfp_mm_class_register(void)
{
    int err;

    err = class_register(&mm_class);
    if (err)
        goto class_register_failed;

    return 0;

class_register_failed:
    return err;
}
EXPORT_SYMBOL(enyx_hfp_mm_class_register);

void
enyx_hfp_mm_class_unregister(void)
{
    class_unregister(&mm_class);
}
EXPORT_SYMBOL(enyx_hfp_mm_class_unregister);

static loff_t
device_llseek(struct file *filp, loff_t off, int whence)
{
    struct enyx_hfp_mm_usr * dev = filp->private_data;
    loff_t newpos;

    if (down_interruptible(&dev->lock))
        return -ERESTARTSYS;

    switch(whence) {
    case 0: /* SEEK_SET */
        newpos = off;
        break;

    case 1: /* SEEK_CUR */
        newpos = filp->f_pos + off;
        break;

    case 2: /* SEEK_END */
        newpos = dev->size;
        break;

    default:
        newpos = -EINVAL;
        break;
    }

    if (newpos < 0 || newpos > dev->size)
        newpos = -EINVAL;

    if (newpos >= 0)
        filp->f_pos = newpos;

    up(&dev->lock);

    return newpos;
}

/**
 *
 */
static int
device_open(struct inode *inode, struct file *filp)
{
    struct enyx_hfp_mm_usr * dev = container_of(inode->i_cdev,
                                                struct enyx_hfp_mm_usr,
                                                cdev);
    int err;

    dev_dbg(&dev->device, "Opening mm device\n");

    if (! enyx_hfp_device_get(dev->parent)) {
        err = -ENODEV;
        goto enyx_hfp_device_get_failed;
    }

    filp->private_data = dev;

    err = 0;

enyx_hfp_device_get_failed:
    return err;
}

/**
 *
 */
static int
device_release(struct inode *inode, struct file *filp)
{
    struct enyx_hfp_mm_usr * dev = filp->private_data;

    dev_dbg(&dev->device, "Releasing mm device\n");

    enyx_hfp_device_put(dev->parent);

    return 0;
}

/**
 *  device_device device mmap syscall implementation.
 *
 *  This function maps the whole MM space
 *  to userland.
 *
 *  @filp: The file instance representing the device.
 *  @vma: The kernel vma to fill.
 */
static int
device_mmap(struct file *filp, struct vm_area_struct *vma)
{
    struct enyx_hfp_mm_usr * dev = filp->private_data;
    unsigned long io_offset = vma->vm_pgoff << PAGE_SHIFT,
                  io_start = dev->phys_addr + io_offset,
                  vm_size = vma->vm_end - vma->vm_start;
    int err;

    if (down_interruptible(&dev->lock)) {
        err = -ERESTARTSYS;
        goto lock_failed;
    }

    HFP_VM_FLAGS_SET(vma, VM_RAND_READ);

    if (vm_size > (dev->size - io_offset)) {
        err = -EINVAL;
        goto err_invalid_arg;
    }

    err = io_remap_pfn_range(vma,
                             vma->vm_start,
                             io_start >> PAGE_SHIFT,
                             vm_size,
                             pgprot_noncached(vma->vm_page_prot));

err_invalid_arg:
    up(&dev->lock);
lock_failed:
    return err;
}

static const struct file_operations device_file_ops = {
    .owner = THIS_MODULE,
    .open = device_open,
    .release = device_release,
    .llseek = device_llseek,
    .mmap = device_mmap,
};

/**
 *  dev_attr_bar_size: The device bar_size attribute.
 */
static ssize_t
size_show(struct device * device,
          struct device_attribute * attr,
          char * buf)
{
    struct enyx_hfp_mm_usr * dev = to_enyx_hfp_mm_usr(device);
    return scnprintf(buf, PAGE_SIZE, "%lu\n", dev->size);
}

static struct device_attribute dev_attr_size = __ATTR_RO(size);

/**
 *  dev_attr_bar_index: The device bar_index attribute.
 */
static ssize_t
index_show(struct device * device,
           struct device_attribute * attr,
           char * buf)
{
    struct enyx_hfp_mm_usr * dev = to_enyx_hfp_mm_usr(device);
    return scnprintf(buf, PAGE_SIZE, "%u\n", dev->index);
}

static struct device_attribute dev_attr_index = __ATTR_RO(index);

/**
 *  device_device_attrs: Device attributes.
 */
static struct attribute * device_attrs[] = {
    &dev_attr_size.attr,
    &dev_attr_index.attr,
    NULL,
};

/**
 *  device_device_group: Device attributes group.
 */
static const struct attribute_group device_group = {
    .attrs = device_attrs,
};

/**
 *  device_device_groups: Device attributes groups.
 */
static const struct attribute_group * device_groups[] = {
    &device_group,
    NULL
};

void
destroy_enyx_hfp_mm_usr(struct enyx_hfp_mm_usr * dev)
{
    device_unregister(&dev->device);
}
EXPORT_SYMBOL(destroy_enyx_hfp_mm_usr);

static void
release_device(struct device * device)
{
    struct enyx_hfp_mm_usr * dev = to_enyx_hfp_mm_usr(device);

    dev_dbg(&dev->device, "Destroying\n");

    cdev_del(&dev->cdev);
    clear_bit(dev->index, device_used);
    enyx_hfp_char_device_put(dev->region, dev->cdev.dev);
    kfree(dev);
}

static const struct device_type device_type = {
    .name = "enyx_hfp_mm",
    .groups = device_groups,
    .release = release_device,
};

struct enyx_hfp_mm_usr *
__create_enyx_hfp_mm_usr(struct enyx_hfp_device * parent,
                    struct enyx_hfp_char_device_region * region,
                    struct module * owner)
{
    struct enyx_hfp_mm_usr * dev;
    dev_t dev_id;

    dev = kzalloc(sizeof(*dev), GFP_KERNEL);
    if (! dev) {
        dev_err(&parent->device, "Can't allocate mm\n");
        goto kzalloc_failed;
    }

    dev->parent = parent;

    dev->size = parent->size;
    dev->phys_addr = parent->bus->phys_addr + parent->bus_offset;
    if (dev->phys_addr % PAGE_SIZE) {
        dev_err(&parent->device,
                "Can't export non-aligned mm at %pap\n", &dev->phys_addr);
        goto align_failed;
    }

    sema_init(&dev->lock, 1);

    dev->region = region;
    dev_id = enyx_hfp_char_device_get(region);

    dev->index = find_first_zero_bit(device_used, sizeof(device_used));
    set_bit(dev->index, device_used);

    cdev_init(&dev->cdev, &device_file_ops);
    dev->cdev.owner = owner;
    if (cdev_add(&dev->cdev, dev_id, 1) < 0) {
        dev_err(&parent->device,
                "Can't add the mm char device %u:%u\n",
                MAJOR(dev_id), MINOR(dev_id));
        goto cdev_add_failed;
    }

    dev->device.devt = dev->cdev.dev;
    dev->device.parent = &parent->device;
    dev->device.type = &device_type;
    dev->device.class = &mm_class;
    if (dev_set_name(&dev->device, "enyx_hfp_mm%u", dev->index) < 0) {
        dev_err(&parent->device, "Can't set enyx_hfp mm device name\n");
        goto dev_set_name_failed;
    }

    if (device_register(&dev->device) < 0) {
        dev_err(&parent->device, "Can't register enyx_hfp mm device\n");
        goto device_register_failed;
    }

    dev_dbg(&dev->device, "Created\n");

    return dev;

device_register_failed:
    put_device(&dev->device);
    /* The device will be garbage collected */
    return NULL;
dev_set_name_failed:
    cdev_del(&dev->cdev);
cdev_add_failed:
    clear_bit(dev->index, device_used);
    enyx_hfp_char_device_put(region, dev_id);
align_failed:
    kfree(dev);
kzalloc_failed:
    return NULL;
}
EXPORT_SYMBOL(__create_enyx_hfp_mm_usr);

