/*
 * HFP bus implementation.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "enyx_hfp_bus.h"

#include <linux/version.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/bitops.h>
#include <linux/delay.h>

#include "enyx_hfp_common.h"
#include "enyx_hfp_universal_compatibility.h"


#define to_enyx_hfp_device(n) \
    container_of(n, struct enyx_hfp_device, device)

#if LINUX_VERSION_CODE >= KERNEL_VERSION(5, 14, 0)
#    define USE_5_15_BUS_REMOVE
#endif

static DECLARE_BITMAP(bus_used, 128);

static int
match_enyx_hfp_id(const struct enyx_hfp_device_id * expected,
             const struct enyx_hfp_device_id * actual)
{
    if (expected->hardware_id != actual->hardware_id)
        return 0;

    if (expected->version_major != HFP_DEVICE_ANY_VERSION &&
        expected->version_major != actual->version_major)
        return 0;

    if (expected->version_minor != HFP_DEVICE_ANY_VERSION &&
        expected->version_minor != actual->version_minor)
        return 0;

    return 1;
}

/**
 *  @note When matching, the device framework will bind
 *        the driver to the device, allow later retrieval
 *        from bus callbacks.
 */
HFP_BUS_MATCH_SIGNATURE
{
    struct enyx_hfp_device * enyx_hfp_device = to_enyx_hfp_device(dev);
    const struct enyx_hfp_device_id * i = to_enyx_hfp_driver(driver)->id_table;

    /* When driver_override is set, only bind to the matching driver and always match it */
    if (enyx_hfp_device->driver_override)
        return !strcmp(enyx_hfp_device->driver_override, to_enyx_hfp_driver(driver)->name);

    for (; i->hardware_id; ++i)
        if (match_enyx_hfp_id(i, &enyx_hfp_device->id))
            return 1;

    return 0;
}

static int
bus_probe(struct device * dev)
{
    struct enyx_hfp_device * enyx_hfp_device = to_enyx_hfp_device(dev);
    struct enyx_hfp_driver * enyx_hfp_driver = to_enyx_hfp_driver(dev->driver);
    int err = 0;

    get_device(dev);

    if (enyx_hfp_driver->probe) {
        err = enyx_hfp_driver->probe(enyx_hfp_device);
        if (err < 0)
            put_device(dev);
    }

    return err;
}

#ifdef USE_5_15_BUS_REMOVE
static void
#else
static int
#endif
bus_remove(struct device * dev)
{
    struct enyx_hfp_device * enyx_hfp_device = to_enyx_hfp_device(dev);
    struct enyx_hfp_driver * enyx_hfp_driver = to_enyx_hfp_driver(dev->driver);

    if (enyx_hfp_driver->remove)
        enyx_hfp_driver->remove(enyx_hfp_device);

    put_device(dev);

#ifndef USE_5_15_BUS_REMOVE
    return 0;
#endif
}

/**
 *
 */
static ssize_t
modalias_show(struct device * dev,
              struct device_attribute * attr,
              char * buf)
{
    struct enyx_hfp_device * enyx_hfp = to_enyx_hfp_device(dev);
    return scnprintf(buf, PAGE_SIZE, "%s\n", enyx_hfp->modalias);
}

static DEVICE_ATTR_RO(modalias);

/**
 *
 */
static ssize_t
bus_offset_show(struct device * dev,
                struct device_attribute * attr,
                char * buf)
{
    struct enyx_hfp_device * enyx_hfp = to_enyx_hfp_device(dev);
    return scnprintf(buf, PAGE_SIZE, "%llx\n", enyx_hfp->bus_offset);
}

static DEVICE_ATTR_RO(bus_offset);

/**
 *
 */
static ssize_t
id_show(struct device * dev,
        struct device_attribute * attr,
        char * buf)
{
    struct enyx_hfp_device * enyx_hfp = to_enyx_hfp_device(dev);
    return scnprintf(buf, PAGE_SIZE, "%hu\n", enyx_hfp->id.hardware_id);
}

static DEVICE_ATTR_RO(id);

/**
 *
 */
static ssize_t
version_major_show(struct device * dev,
                   struct device_attribute * attr,
                   char * buf)
{
    struct enyx_hfp_device * enyx_hfp = to_enyx_hfp_device(dev);
    return scnprintf(buf, PAGE_SIZE, "%hhu\n", enyx_hfp->id.version_major);
}

static DEVICE_ATTR_RO(version_major);

/**
 *
 */
static ssize_t
version_minor_show(struct device * dev,
                   struct device_attribute * attr,
                   char * buf)
{
    struct enyx_hfp_device * enyx_hfp = to_enyx_hfp_device(dev);
    return scnprintf(buf, PAGE_SIZE, "%hhu\n", enyx_hfp->id.version_minor);
}

static DEVICE_ATTR_RO(version_minor);

/**
 *
 */
static ssize_t
product_version_show(struct device * dev,
              struct device_attribute * attr,
              char * buf)
{
    struct enyx_hfp_device * enyx_hfp = to_enyx_hfp_device(dev);
    uint8_t const major =  enyx_hfp->id.revision >> 10;
    uint8_t const minor = (enyx_hfp->id.revision >> 4) & 0x3F;
    uint8_t const bugfix = enyx_hfp->id.revision & 0xF;

    if (enyx_hfp->id.revision == 0)
        return scnprintf(buf, PAGE_SIZE, "\n");

    return scnprintf(buf, PAGE_SIZE, "%hhu.%hhu.%hhu\n", major, minor, bugfix);
}

static DEVICE_ATTR_RO(product_version);

/**
 *
 */
static ssize_t
driver_override_store(struct device *dev,
                     struct device_attribute *attr,
                     const char *buf, size_t count)
{
    struct enyx_hfp_device * enyx_hfp = to_enyx_hfp_device(dev);
    char *driver_override, *old, *cp;

    if (count >= (PAGE_SIZE - 1))
        return -EINVAL;

    driver_override = kstrndup(buf, count, GFP_KERNEL);
    if (!driver_override)
        return -ENOMEM;

    cp = strchr(driver_override, '\n');
    if (cp)
        *cp = '\0';

    old = enyx_hfp->driver_override;
    if (!driver_override[0]) {
        enyx_hfp->driver_override = driver_override;
    } else {
        kfree(driver_override);
        enyx_hfp->driver_override = NULL;
    }

    kfree(old);

    return count;
}

static ssize_t
driver_override_show(struct device *dev,
                    struct device_attribute *attr, char *buf)
{
    struct enyx_hfp_device * enyx_hfp = to_enyx_hfp_device(dev);

    return scnprintf(buf, PAGE_SIZE, "%s\n", enyx_hfp->driver_override);
}

static DEVICE_ATTR_RW(driver_override);

/**
 *
 */
HFP_BUS_UEVENT_SIGNATURE
{
    struct enyx_hfp_device * enyx_hfp = to_enyx_hfp_device(dev);

    add_uevent_var(env, "MODALIAS=%s%s",
                   MODULE_ALIAS_HFP_PREFIX, enyx_hfp->modalias);
    return 0;
}

static struct attribute * device_attrs[] = {
    &dev_attr_modalias.attr,
    &dev_attr_bus_offset.attr,
    &dev_attr_id.attr,
    &dev_attr_version_major.attr,
    &dev_attr_version_minor.attr,
    &dev_attr_product_version.attr,
    &dev_attr_driver_override.attr,
    NULL,
};

ATTRIBUTE_GROUPS(device);

static struct bus_type bus_type = {
    .name = "enyx_hfp",
    .uevent = bus_uevent,
    .match = bus_match,
    .probe = bus_probe,
    .remove = bus_remove,
    .dev_groups = device_groups,
};

int
enyx_hfp_wait_until_val_is(struct enyx_hfp_device * dev, size_t offset, uint8_t val)
{
    uint32_t tries = 10;

    while (enyx_hfp_read8(dev, offset) != val && tries) {
        cpu_relax();
        udelay(10);
        -- tries;
    }

    return -(enyx_hfp_read8(dev, offset) != val);
}
EXPORT_SYMBOL(enyx_hfp_wait_until_val_is);

size_t
enyx_hfp_read_str_unlocked(struct enyx_hfp_device * dev,
                      size_t str_reg,
                      size_t str_index_reg,
                      size_t ready_reg,
                      char * buffer, size_t buffer_capacity)
{
    size_t str_index = 0;
    size_t chunk_size = 4;
    size_t offset = 0;
    union
    {
        char str[5];
        uint32_t i;
    } chunk = { {0} };

    while (chunk_size == 4) {
        enyx_hfp_write16(dev, str_index_reg, str_index);
        if (enyx_hfp_wait_until_val_is(dev, ready_reg, 1))
            return 0;

        chunk.i = enyx_hfp_read32(dev, str_reg);

        chunk_size = strlen(strncpy(buffer + offset,
                                    chunk.str,
                                    buffer_capacity));

        offset += chunk_size;
        buffer_capacity -= chunk_size;

        ++ str_index;
    }

    return offset;
}
EXPORT_SYMBOL(enyx_hfp_read_str_unlocked);

size_t enyx_hfp_read32_locked_v0_2(struct enyx_hfp_device *dev,
        size_t read_reg,
        size_t page_reg,
        size_t ready_reg,
        uint16_t page_id)
{
    size_t count = -1ULL;

    if (down_interruptible(&dev->lock))
        goto lock_failed;

    enyx_hfp_write8(dev, page_reg, page_id);
    if (enyx_hfp_wait_until_val_is(dev, ready_reg, 1) < 0)
    {
        dev_err(&dev->device, "info page failed %u\n", page_id);
        goto page_select_failed;
    }

    count = enyx_hfp_read32(dev, read_reg);

page_select_failed:
    up(&dev->lock);
lock_failed:
    return count;
}
EXPORT_SYMBOL(enyx_hfp_read32_locked_v0_2);

size_t enyx_hfp_read16_locked_v0_2(struct enyx_hfp_device *dev,
        size_t read_reg,
        size_t page_reg,
        size_t ready_reg,
        uint16_t page_id)
{
    size_t count = -1ULL;

    if (down_interruptible(&dev->lock))
        goto lock_failed;

    enyx_hfp_write8(dev, page_reg, page_id);
    if (enyx_hfp_wait_until_val_is(dev, ready_reg, 1) < 0)
    {
        dev_err(&dev->device, "info page failed %u\n", page_id);
        goto page_select_failed;
    }

    count = enyx_hfp_read16(dev, read_reg);

page_select_failed:
    up(&dev->lock);
lock_failed:
    return count;
}
EXPORT_SYMBOL(enyx_hfp_read16_locked_v0_2);

void
enyx_hfp_bus_for_each_safe(struct enyx_hfp_bus * bus,
                      void (*cb)(struct enyx_hfp_device *, void *),
                      void * opaque)
{
    struct enyx_hfp_device * cur, * sav;

    list_for_each_entry_safe(cur, sav, &bus->devices, bus_node)
        (*cb)(cur, opaque);
}
EXPORT_SYMBOL(enyx_hfp_bus_for_each_safe);

static void
release_device(struct device * dev)
{
    struct enyx_hfp_device * hfp_device = to_enyx_hfp_device(dev);

    dev_dbg(&hfp_device->device, "Destroying\n");

    kfree(hfp_device->driver_override);

    list_del(&hfp_device->node);
    kfree(hfp_device);
}

static const struct device_type device_type = {
    .name = "enyx_hfp_device",
    .release = release_device,
};

static struct enyx_hfp_device *
initialize_enyx_hfp_device(const struct enyx_hfp_device_id * id,
                           struct enyx_hfp_bus * bus,
                           uint64_t bus_offset,
                           size_t size,
                           struct enyx_hfp_device * parent)
{
    struct enyx_hfp_device * hfp_device;

    dev_dbg(&bus->device,
            "Creating enyx_hfp_device at bus offset %#llx-%#llx\n",
            bus_offset, bus_offset + size);

    hfp_device = kzalloc(sizeof(*hfp_device), GFP_KERNEL);
    if (! hfp_device)
    {
        dev_err(&bus->device, "Can't allocated hfp_device device\n");
        goto kzalloc_failed;
    }

    memcpy(&hfp_device->id, id, sizeof(hfp_device->id));
    if (snprintf(hfp_device->modalias, sizeof(hfp_device->modalias),
                 "%huma%hhumi%hhu",
                 hfp_device->id.hardware_id,
                 hfp_device->id.version_major,
                 hfp_device->id.version_minor) >= sizeof(hfp_device->modalias))
    {
        dev_err(&bus->device, "Can't generate modalias\n");
        goto snprintf_failed;
    }

    hfp_device->bus_offset = bus_offset;
    hfp_device->size = size;
    hfp_device->bus = bus;
    hfp_device->device.parent = parent ? &parent->device : &bus->device;
    hfp_device->device.bus = &bus_type;
    hfp_device->device.type = &device_type;
    if (dev_set_name(&hfp_device->device,
                     "%u:%llx:%hu:%hhu.%hhu",
                     bus->index,
                     hfp_device->bus_offset,
                     hfp_device->id.hardware_id,
                     hfp_device->id.version_major,
                     hfp_device->id.version_minor) < 0) {
        dev_err(&bus->device, "Can't set hfp_device device name\n");
        goto dev_set_name_failed;
    }

    sema_init(&hfp_device->lock, 1);

    device_initialize(&hfp_device->device);

    INIT_LIST_HEAD(&hfp_device->bus_node);
    list_add_tail(&hfp_device->bus_node, &bus->devices);

    INIT_LIST_HEAD(&hfp_device->children);
    INIT_LIST_HEAD(&hfp_device->node);
    if (parent)
        list_add_tail(&hfp_device->node, &parent->children);

    dev_dbg(&hfp_device->device, "Created\n");

    return hfp_device;

dev_set_name_failed:
snprintf_failed:
    kfree(hfp_device);
kzalloc_failed:
    return NULL;
}

static void
destroy_enyx_hfp_device(struct enyx_hfp_device * hfp_device)
{
    list_del(&hfp_device->bus_node);
    device_unregister(&hfp_device->device);
}

struct enyx_hfp_device *
enyx_hfp_get_device_parent(struct enyx_hfp_device * dev)
{
    struct device * d = &dev->device;
    do {
        d = d->parent;
    } while (d && d->type != &device_type);

    return d ? to_enyx_hfp_device(d) : NULL;
}
EXPORT_SYMBOL(enyx_hfp_get_device_parent);

int
enyx_hfp_for_each_child(struct enyx_hfp_device * device,
                        int (*fn)(struct enyx_hfp_device * dev, void * opaque),
                        void * opaque)
{
    struct enyx_hfp_device * cur, * sav;
    int ret;

    list_for_each_entry_safe(cur, sav, &device->children, node)
        if ((ret = fn(cur, opaque)))
            goto fn_failed;

    ret = 0;

fn_failed:
    return ret;
}
EXPORT_SYMBOL(enyx_hfp_for_each_child);

struct find_depth_first_data
{
    const struct enyx_hfp_device_id * expected_id;
    size_t skip_count;
    struct enyx_hfp_device * device;
};

static int
find_depth_first(struct enyx_hfp_device * device,
                 void * opaque)
{
    struct find_depth_first_data * data = opaque;

    if (match_enyx_hfp_id(data->expected_id, &device->id)) {
        if (data->skip_count == 0) {
            data->device = device;
            return 1;
        }

        -- data->skip_count;
    }

    return enyx_hfp_for_each_child(device, find_depth_first, data);
}

struct enyx_hfp_device *
enyx_hfp_find_depth_first(struct enyx_hfp_device * device,
                     const struct enyx_hfp_device_id * id,
                     size_t device_index)
{
    struct find_depth_first_data data = {
        .expected_id = id,
        .skip_count = device_index,
    };

    find_depth_first(device, &data);

    return data.device;
}
EXPORT_SYMBOL(enyx_hfp_find_depth_first);

static void
release_bus(struct device * dev)
{
    struct enyx_hfp_bus * bus = to_enyx_hfp_bus(dev);

    clear_bit(bus->index, bus_used);
    kfree(bus);
}

static struct class bus_class = {
    .name = "enyx_hfp_bus",
    HFP_CLASS_OWNER_FIELD
};

int
enyx_hfp_bus_type_register(void)
{
    int err;

    err = class_register(&bus_class);
    if (err)
        goto class_register_failed;

    err = bus_register(&bus_type);
    if (err)
        goto bus_register_failed;

    return 0;

bus_register_failed:
    class_unregister(&bus_class);
class_register_failed:
    return err;
}

void
enyx_hfp_bus_type_unregister(void)
{
    bus_unregister(&bus_type);
    class_unregister(&bus_class);
}

static void
get_device_id_from_r0(uint32_t r0,
                      struct enyx_hfp_device_id * id)
{
    id->version_minor = r0;
    id->version_major = r0 >> 8;
    id->hardware_id = r0 >> 16;
}

static int
get_child_size_from_r1(uint32_t r1,
                       size_t * child_size,
                       uint8_t * child_count)
{
    const uint8_t child_addr_width = (r1 >> 8);

    if (child_addr_width > U8_MAX - HFP_BUS_ADDR_SHIFT)
        goto metadata_check_failed;

    *child_size = 1ULL << (child_addr_width + HFP_BUS_ADDR_SHIFT);

    *child_count = r1;

    if (*child_count == 0)
        goto metadata_check_failed;

    return 0;

metadata_check_failed:
    return -EIO;
}

static uint16_t
get_revision_from_r1(uint32_t r1)
{
    return r1 >> 16;
}

static int
initialize_enyx_hfp_devices(struct enyx_hfp_bus * bus,
                            uint64_t bus_offset,
                            size_t size,
                            struct enyx_hfp_device * parent)
{
    size_t child_size;
    uint8_t i, child_count;
    struct enyx_hfp_device_id id;
    struct enyx_hfp_device * device;

    if (get_child_size_from_r1(readl(bus->virt_addr + bus_offset + 4),
                               &child_size, &child_count) < 0) {
        dev_err(&bus->device,
                "Skipping device with corrupted metadata at %llx\n",
                bus_offset);
        dev_err(&bus->device,
                "Can't retrieve children metadata\n");
        return 0;
    }

    if (child_size * child_count > size) {
        dev_err(&bus->device,
                "Skipping device with corrupted metadata at %llx\n",
                bus_offset);
        dev_err(&bus->device,
                "child_size (%zu) * child_count (%hhu) > parent_size (%zu)\n",
                child_size, child_count, size);
        return 0;
    }

    get_device_id_from_r0(readl(bus->virt_addr + bus_offset), &id);
    id.revision = get_revision_from_r1(readl(bus->virt_addr + bus_offset + 4));

    dev_dbg(&bus->device, "Found device %hu:%hhu:%hhu at %llx "
                          "with %hhu children of %zu bytes\n",
            id.hardware_id, id.version_major, id.version_minor, bus_offset,
            child_count - 1, child_size);

    device = initialize_enyx_hfp_device(&id, bus, bus_offset, size, parent);
    if (! device)
        goto initialize_enyx_hfp_device_failed;

    for (i = 1; i != child_count; ++i) {
        bus_offset += child_size;
        if (initialize_enyx_hfp_devices(bus, bus_offset, child_size, device) < 0)
            goto initialize_enyx_hfp_devices_failed;
    }

    return 0;

initialize_enyx_hfp_devices_failed:
initialize_enyx_hfp_device_failed:
    return -EIO;
}

static size_t
get_bus_size(phys_addr_t phys_addr, struct device * dev)
{
    size_t child_size;
    uint8_t child_count;
    size_t result = 0;

    uint8_t __iomem * virt_addr = ioremap(phys_addr, 8);
    if (! virt_addr) {
        dev_err(dev, "Can't remap bus memory\n");
        goto ioremap_failed;
    }

    if (get_child_size_from_r1(readl(virt_addr + 4),
                               &child_size, &child_count) < 0) {
        dev_err(dev, "Bus root metadata are corrupted\n");
        goto get_child_size_from_r1_failed;
    }

    result = child_count * child_size;

get_child_size_from_r1_failed:
    iounmap(virt_addr);
ioremap_failed:
    return result;
}

static int
create_enyx_hfp_devices(struct enyx_hfp_bus * bus)
{
    struct enyx_hfp_device *cur, *sav;
    size_t bus_size;

    bus_size = get_bus_size(bus->phys_addr, &bus->device);
    if (! bus_size)
        goto get_bus_size_failed;

    /* Map the MM address space within the kernel address space */
    bus->virt_addr = ioremap(bus->phys_addr, bus_size);
    if (! bus->virt_addr) {
        dev_err(&bus->device, "Can't remap bus memory\n");
        goto ioremap_failed;
    }

    dev_dbg(&bus->device,
            "mm address space located spanning 0x0-0x%zx\n", bus_size);

    /* Perform creation in 2 steps in order to ensure
     * that a registered device knowns its children at
     * driver binding:
     * 1. Discover devices and initialize them */
    if (initialize_enyx_hfp_devices(bus, 0, bus_size, NULL) < 0)
        goto initialized_failed;

    /* 2. Register devices */
    list_for_each_entry(cur, &bus->devices, bus_node)
        if (device_add(&cur->device) < 0) {
            dev_err(&cur->device, "Can't register hfp_device device\n");
            goto device_add_failed;
        }

    return 0;

device_add_failed:
    /* Destroy registered devices
     * (missing list_for_each_entry_safe_continue_reverse here) */
    cur = list_prev_entry(cur, bus_node);
    for (; &cur->bus_node != &bus->devices; ) {
        sav = list_prev_entry(cur, bus_node);
        destroy_enyx_hfp_device(cur);
        cur = sav;
    }
initialized_failed:
    /* Destroy unregistered devices */
    list_for_each_entry_safe(cur, sav, &bus->devices, bus_node)
        release_device(&cur->device);
    iounmap(bus->virt_addr);
ioremap_failed:
get_bus_size_failed:
    return -EIO;
}

static void
destroy_enyx_hfp_devices(struct enyx_hfp_bus * bus)
{
    struct enyx_hfp_device * cur, * sav;

    list_for_each_entry_safe_reverse(cur, sav, &bus->devices, bus_node)
        destroy_enyx_hfp_device(cur);

    list_del_init(&bus->devices);

    iounmap(bus->virt_addr);
}

static int
count_devices_used(struct enyx_hfp_bus * bus, size_t * devices_used)
{
    int err;

    err = down_interruptible(&bus->devices_lock);
    if (err < 0)
        goto lock_failed;

    *devices_used = bus->devices_used;

    up(&bus->devices_lock);
lock_failed:
    return err;
}

int
clear_enyx_hfp_bus(struct enyx_hfp_bus * bus)
{
    int err;

    err = down_interruptible(&bus->devices_lock);
    if (err < 0)
        goto lock_failed;

    if (list_empty(&bus->devices)) {
        dev_dbg(&bus->device, "Clearing empty bus\n");
        goto already_empty;
    }

    if (bus->devices_used) {
        err = -EBUSY;
        goto bus_used;
    }

    destroy_enyx_hfp_devices(bus);

    dev_dbg(&bus->device, "Cleared\n");

bus_used:
already_empty:
    up(&bus->devices_lock);
lock_failed:
    return err;
}
EXPORT_SYMBOL(clear_enyx_hfp_bus);

int
scan_enyx_hfp_bus(struct enyx_hfp_bus * bus)
{
    int err;

    err = down_interruptible(&bus->devices_lock);
    if (err < 0)
        goto lock_failed;

    if (! list_empty(&bus->devices)) {
        dev_err(&bus->device, "Bus is already populated\n");
        goto already_populated;
    }

    err = create_enyx_hfp_devices(bus);
    if (err >= 0)
        dev_dbg(&bus->device, "Scanned\n");

already_populated:
    up(&bus->devices_lock);
lock_failed:
    return err;
}
EXPORT_SYMBOL(scan_enyx_hfp_bus);

int
get_enyx_hfp_bus_version(struct enyx_hfp_bus * bus,
                         uint8_t * major,
                         uint8_t * minor,
                         uint8_t * patch)
{
    int err = 0;
    struct enyx_hfp_device * device;

    if (!bus || ! major || ! minor || ! patch)
        return -EINVAL;

    err = down_interruptible(&bus->devices_lock);
    if (err < 0)
        goto lock_failed;

    if (list_empty(&bus->devices)) {
        dev_err(&bus->device, "Cannot get version of empty bus\n");
        err = -ENOENT;
        goto nobus;
    }
    device = list_first_entry(&bus->devices,
                              struct enyx_hfp_device,
                              bus_node);

    *major =  device->id.revision >> 10;
    *minor = (device->id.revision >> 4) & 0x3F;
    *patch = device->id.revision & 0xF;

nobus:
    up(&bus->devices_lock);
lock_failed:
    return err;
}
EXPORT_SYMBOL(get_enyx_hfp_bus_version);

/**
 *  dev_attr_index: The device index attribute.
 */
static ssize_t
index_show(struct device * dev,
               struct device_attribute * attr,
               char * buf)
{
    struct enyx_hfp_bus * bus = to_enyx_hfp_bus(dev);
    return scnprintf(buf, PAGE_SIZE, "%u\n", bus->index);
}

static DEVICE_ATTR_RO(index);

/**
 *  dev_attr_used_count: The device used_count attribute.
 */
static ssize_t
devices_used_count_show(struct device * dev,
                        struct device_attribute * attr,
                        char * buf)
{
    struct enyx_hfp_bus * bus = to_enyx_hfp_bus(dev);
    size_t devices_used;
    int err;

    err = count_devices_used(bus, &devices_used);
    if (err < 0)
        return err;

    return scnprintf(buf, PAGE_SIZE, "%zu\n", devices_used);
}

static DEVICE_ATTR_RO(devices_used_count);

static ssize_t
scan_store(struct device *dev,
           struct device_attribute *attr,
           const char *buf, size_t count)
{
    struct enyx_hfp_bus * bus = to_enyx_hfp_bus(dev);
    int err;

    if (count >= (PAGE_SIZE - 1))
        return -EINVAL;

    if (! sysfs_streq(buf, "1"))
        return -EINVAL;

    err = scan_enyx_hfp_bus(bus);
    if (err < 0)
        count = err;

    return count;
}

static DEVICE_ATTR_WO(scan);

static ssize_t
clear_store(struct device *dev,
            struct device_attribute *attr,
            const char *buf, size_t count)
{
    struct enyx_hfp_bus * bus = to_enyx_hfp_bus(dev);
    int err;

    if (count >= (PAGE_SIZE - 1))
        return -EINVAL;

    if (! sysfs_streq(buf, "1"))
        return -EINVAL;

    err = clear_enyx_hfp_bus(bus);
    if (err < 0)
        count = err;

    return count;
}

static DEVICE_ATTR_WO(clear);

static ssize_t
bus_version_show(struct device * dev,
                 struct device_attribute *attr,
                 char * buf)
{
    struct enyx_hfp_bus * bus = to_enyx_hfp_bus(dev);
    uint8_t major, minor, patch;

    if (get_enyx_hfp_bus_version(bus, &major, &minor, &patch) < 0)
        return scnprintf(buf, PAGE_SIZE, "\n");

    return scnprintf(buf, PAGE_SIZE, "%hhu.%hhu.%hhu\n", major, minor, patch);

    return 0;
}

static DEVICE_ATTR_RO(bus_version);

static ssize_t
bus_alias_store(struct device * dev,
                struct device_attribute *attr,
                const char * buf, size_t count)
{
    struct enyx_hfp_bus * bus = to_enyx_hfp_bus(dev);
    char *cp;

    if (count >= (BUSNAMESIZE - 1) || count == 0)
        return -EINVAL;

    scnprintf(bus->bus_alias, BUSNAMESIZE, "%s", buf);

    cp = strchr(bus->bus_alias, '\n');
    if (cp)
        *cp = '\0';

    return count;
}

static ssize_t
bus_alias_show(struct device * dev,
               struct device_attribute * attr,
               char * buf)
{
    struct enyx_hfp_bus * bus = to_enyx_hfp_bus(dev);

    return scnprintf(buf, PAGE_SIZE, "%s", bus->bus_alias);
}

static DEVICE_ATTR_RW(bus_alias);

/**
 *  bus_device_attrs: Device attributes.
 */
static struct attribute * bus_attrs[] = {
    &dev_attr_index.attr,
    &dev_attr_devices_used_count.attr,
    &dev_attr_scan.attr,
    &dev_attr_clear.attr,
    &dev_attr_bus_version.attr,
    &dev_attr_bus_alias.attr,
    NULL,
};

ATTRIBUTE_GROUPS(bus);

/**
 *  bus_device_type: Device metadata.
 */
static const struct device_type bus_device_type = {
    .name = "enyx_hfp_bus",
    .groups = bus_groups,
    .release = release_bus,
};

struct enyx_hfp_bus *
__create_enyx_hfp_bus(phys_addr_t phys_addr,
                 struct enyx_hfp_irq_list * irq_list,
                 struct device * parent,
                 struct module * owner)
{
    struct enyx_hfp_bus * bus;

    dev_dbg(parent, "Creating enyx_hfp_bus at %pa\n", &phys_addr);

    bus = kzalloc(sizeof(*bus), GFP_KERNEL);
    if (! bus) {
        dev_err(parent, "Can't allocate enyx_hfp bus\n");
        goto kzalloc_failed;
    }

    bus->irq_list = irq_list;

    bus->phys_addr = phys_addr;
    INIT_LIST_HEAD(&bus->devices);

    bus->index = find_first_zero_bit(bus_used, sizeof(bus_used));
    set_bit(bus->index, bus_used);

    scnprintf(bus->bus_alias, BUSNAMESIZE, "board%u", bus->index);

    bus->owner = owner;
    bus->device.parent = parent;
    bus->device.type = &bus_device_type;
    bus->device.class = &bus_class;

    if (dev_set_name(&bus->device, "enyx_hfp_bus%u", bus->index) < 0) {
        dev_err(parent, "Can't set enyx_hfp bus name\n");
        goto dev_set_name_failed;
    }

    if (device_register(&bus->device) < 0) {
        dev_err(parent, "Can't register enyx_hfp bus\n");
        goto device_register_failed;
    }

    sema_init(&bus->devices_lock, 1);

    dev_dbg(&bus->device, "Created\n");

    return bus;

device_register_failed:
    /* The device will be garbage collected */
    put_device(&bus->device);
    return NULL;
dev_set_name_failed:
    clear_bit(bus->index, bus_used);
    kfree(bus);
kzalloc_failed:
    return NULL;
}
EXPORT_SYMBOL(__create_enyx_hfp_bus);

void
destroy_enyx_hfp_bus(struct enyx_hfp_bus * bus)
{
    device_unregister(&bus->device);
}
EXPORT_SYMBOL(destroy_enyx_hfp_bus);

int
__enyx_hfp_register_driver(struct enyx_hfp_driver * driver,
                      struct module * owner,
                      const char * owner_name)
{
    driver->driver.name = driver->name;
    driver->driver.bus = &bus_type;
    driver->driver.owner = owner;
    driver->driver.mod_name = owner_name;

    return driver_register(&driver->driver);
}
EXPORT_SYMBOL(__enyx_hfp_register_driver);

void
enyx_hfp_unregister_driver(struct enyx_hfp_driver * driver)
{
    driver_unregister(&driver->driver);
}
EXPORT_SYMBOL(enyx_hfp_unregister_driver);
