#include <getopt.h>
#include <unistd.h>

#include <cerrno>
#include <cstdlib>
#include <cstring>

#include <iomanip>
#include <sstream>
#include <string>
#include <set>
#include <algorithm>

#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/core.hpp>
#include <enyx/hw/core_tree.hpp>
#include <enyx/hw/mmio.hpp>

#include <enyx/cores/hardware_ids.hpp>
#include <enyx/cores/probes/collector.hpp>
#include <enyx/cores/probes/timestamp.hpp>
#include <enyx/cores/probes/probe.hpp>
#include <enyx/cores/probes/system.hpp>

namespace {

namespace h = enyx::hw;
namespace p = enyx::probes;

using id_range = std::set<p::probe::id>;

struct arguments
{
    std::uint32_t accelerator_id;
    std::uint32_t collector_id;
    enum { INFO, ENABLE, DISABLE } action;
    id_range ids;
};

void
usage(char const * name)
{
    std::cout << "Usage: " << name << " [OPTION]... ACTION\n"
                 "List, enable or disable hardware probes.\n"
                 "\n"
                 "ACTION can be:\n"
                 "  info [RANGE]          Retrieve info on all or specific probe\n"
                 "  enable [RANGE]        Enable all or specific probe\n"
                 "  disable [RANGE]       Disable all or specific probe\n"
                 "\n"
                 "RANGE can be:\n"
                 "  ID                    A probe index (e.g. 1,3,8)\n"
                 "  ID-ID                 A range of probe ID (e.g. 1-4)\n"
                 "  RANGE,RANGE           A list of probe RANGE (e.g. 1,3-5)\n"
                 "Optional arguments:\n"
                 "  -a, --accelerator-id  Select the accelerator (default: 0)\n"
                 "  -c, --collector-id    Select the collector index (default: 0)\n"
                 "  -h, --help            Display this help and exit\n"
              << std::flush;
}

int
parse_id_range(std::string const& range, id_range & result)
{
    std::istringstream buffer{range};
    for (std::string line; std::getline(buffer, line, ',');)
    {
        std::size_t pos;
        auto const first_id = std::stol(line, &pos);

        // Process next range if the current range only contains one integer.
        if (pos == line.size())
        {
            result.insert(first_id);
            continue;
        }

        // Ensure the separator after the current integer is '-'.
        if (line[pos] != '-')
        {
            std::cerr << "Malformed range, expecting '-', got '"
                      << line[pos] << "'.\n"
                      << std::flush;
            return -1;
        }

        // Trim the first integer.
        line.erase(0, pos + 1);
        // Try to parse the second integer.
        auto const second_id = std::stol(line, &pos);

        // Ensure there is not remaining data after the second integer.
        if (pos != line.size())
        {
            std::cerr << "Malformed range, expecting EOS, got '"
                      << line[pos] << "'.\n"
                      << std::flush;
            return -1;
        }

        for (auto i = first_id; i <= second_id; ++i)
            result.insert(i);
    }

    return 0;
}

int
parse_args(int argc, char * argv[], arguments & args)
{
    struct option const options[] = {
        {"accelerator-id", true, nullptr, 'a'},
        {"collector-id", true, nullptr, 'c'},
        {"help", false, nullptr, 'h'},
        {},
    };

    int opt;
    while ((opt = ::getopt_long(argc, argv,
                                "a:c:h", options, nullptr)) != -1)
    {
        switch (opt)
        {
        case 'a':
            args.accelerator_id = std::atoi(optarg);
            break;
        case 'c':
            args.collector_id = std::atoi(optarg);
            break;
        default:
            usage(argv[0]);
            return -1;
        }
    }

    if (optind == argc)
    {
        std::cerr << "Missing ACTION.\n" << std::flush;
        return -1;
    }

    char const * action = argv[optind];
    ++ optind;

    if (std::strcmp(action, "info") == 0)
    {
        args.action = arguments::INFO;
    }
    else if (std::strcmp(action, "enable") == 0)
    {
        args.action = arguments::ENABLE;
    }
    else if (std::strcmp(action, "disable") == 0)
    {
        args.action = arguments::DISABLE;
    }
    else
    {
        std::cerr << "Unknown ACTION \"" << action << "\".\n"
                  << std::flush;
        return -1;
    }

    if (optind != argc)
    {
        if (parse_id_range(argv[optind], args.ids))
            return -1;
        ++ optind;
    }

    if (optind != argc)
    {
        std::cerr << "Unexpected argument \"" << argv[optind] << "\".\n"
                  << std::flush;
        return -1;
    }

    return 0;
}

void
run(arguments & args)
{
    // Construct a filter that requests the accelerator
    h::filter const select_accelerator{h::index{args.accelerator_id}};

    // Find and instantiate the requested accelerator
    auto const accelerator_descriptors = h::enumerate_accelerators(select_accelerator);
    if (accelerator_descriptors.size() != 1)
    {
        std::ostringstream error;
        error << "Expecting exactly 1 accelerator, got "
                  << accelerator_descriptors.size();
        throw std::runtime_error{error.str()};
    }
    h::accelerator accelerator{accelerator_descriptors[0]};

    // Find and instantiate the first mmio from the requested accelerator
    h::filter const select_mmio{h::index{0}};
    auto const mmio_descriptors = accelerator.enumerate_mmios(select_mmio);
    if (mmio_descriptors.size() != 1)
    {
        std::ostringstream error;
        error << "Expecting exactly 1 mmio, got "
              << mmio_descriptors.size();
        throw std::runtime_error{error.str()};
    }

    // Create the probing system
    h::mmio mmio{mmio_descriptors[0]};
    h::core_tree tree{h::enumerate_cores(mmio)};
    h::core root{tree.get_root()};

    // Enumerate collectors
    auto collectors = root.enumerate(enyx::hardware_ids::EVENT_COLLECTOR);
    if (collectors.empty())
        throw std::runtime_error{"Can't find any event collector"};

    if (args.collector_id >= collectors.size())
    {
        std::ostringstream error;
        error << "collector " << args.collector_id << " doesn't exist";
        throw std::runtime_error{error.str()};
    }

    // Retrieve the requested collector
    auto collector = enyx::probes::collector{collectors[args.collector_id]};

    // Print the current collector name
    if (args.action == arguments::INFO)
        std::cout << "collector " << args.collector_id
                  << " \"" << collector.get_name() << "\"" << std::endl;

    auto mtg = enyx::probes::mtg{root};

    // Enumerate current collector's probes
    auto probe_cores = root.enumerate(enyx::hardware_ids::EVENT_PROBE);
    std::vector<enyx::probes::probe> probes;
    enyx::probes::probe::id i = 0;
    for (auto & probe_core: probe_cores) {
        auto probe = enyx::probes::probe{probe_core, i,
                                         mtg.get_ts_format().v()};
        if (probe.get_collector_name() == collector.get_name()) {
            probes.push_back(std::move(probe));
            i++;
        }
    }

    if (probes.empty())
    {
        std::ostringstream error;
        error << "Can't retrieve collector "
              << args.collector_id << "'s probe(s)";
        throw std::runtime_error{error.str()};
    }

    // Iterate over current collector's probes
    for (std::size_t j = 0, f = probes.size(); j != f; ++j)
    {
        auto & probe = probes[j];

        // Filter probe by id if requested
        if (! args.ids.empty() && ! args.ids.count(j))
            continue;

        h::result<void> result;
        switch (args.action)
        {
        case arguments::INFO:
            std::cout << "\tprobe " << j << " \"" << probe.get_name()
                      << "\" using "
                      << int(probe.get_event_format().reference_id_size)
                      << " bytes ref-id, "
                      << probe.get_event_format().ts_format
                      << " timestamp and "
                      << int(probe.get_event_format().custom_info_size)
                      << " bytes custom-info is ";
            if (probe.is_enabled())
                std::cout << "enabled";
            else
                std::cout << "disabled";

            std::cout << std::endl;
            break;

        case arguments::ENABLE:
            result = probe.enable();
            if (! result)
            {
                std::ostringstream error;
                error << "Can't enable the accelerator "
                      << args.collector_id
                      << "'s probe " << j;
                throw std::system_error{result.error(), error.str()};
            }
            break;

        case arguments::DISABLE:
            result = probe.disable();
            if (! result)
            {
                std::ostringstream error;
                error << "Can't disable the accelerator "
                      << args.collector_id << "'s probe " << j;
                throw std::system_error{result.error(), error.str()};
            }
            break;
        }
    }
}

} // anonymous namespace

int main(int argc, char *argv[])
{
    arguments args{};
    if (parse_args(argc, argv, args) < 0)
        return EXIT_FAILURE;

    try
    {
        run(args);
        return EXIT_SUCCESS;
    }
    catch(std::exception const& e)
    {
        std::cerr << e.what() << std::endl;
    }
    catch(...)
    {
        std::cerr << "An unknown error occurred" << std::endl;
    }

    return EXIT_FAILURE;
}

