#include "TcpRuntime.hpp"

#include <cerrno>

#include <algorithm>
#include <atomic>
#include <exception>
#include <functional>
#include <future>
#include <iostream>
#include <memory>
#include <numeric>
#include <sstream>
#include <thread>
#include <vector>

#include <enyx/hw/error.hpp>

#include <enyx/cores/namespace.hpp>

#include "../lib/DeviceDiscovery.hpp"
#include "../lib/SignalHandler.hpp"
#include "DataValidation.hpp"

using namespace std::placeholders;

ENYX_CORES_NAMESPACE_BEGIN

TxState
TcpRuntime::Session::get_tx_state()
{
    if (state.load(std::memory_order_acquire) != ESTABLISHED)
        return TxState::NOT_READY;
    return TxState::READY;
}

std::size_t
TcpRuntime::Session::get_payload_max_size() const
{
    return configuration->mss;
}

TcpRuntime::TcpRuntime(ApplicationConfiguration const & configuration,
                       hw::accelerator & accelerator,
                       hw::core_tree & core_tree)
    : data_sources(get_sources<TcpSource>(accelerator,
                                          configuration.stack_index,
                                           "TCP"))
    , data_sinks(get_sinks<TcpSink>(accelerator,
                                    configuration.stack_index,
                                     "TCP"))
    , stack(find_tcp_core(core_tree, configuration.stack_index))
    , session_count(stack.get_session_count())
    , sessions(session_count)
    , context(std::bind(&TcpRuntime::on_state_change, this, _1, _2, _3),
              std::bind(&TcpRuntime::on_credit_change, this, _1, _2),
              std::bind(&TcpRuntime::on_emi_error, this))
    , events_source(get_tcp_events_source(accelerator,
                                          configuration.stack_index), context)
{ }

void
TcpRuntime::on_state_change(std::uint32_t session_id,
                            tcp::context::state new_state,
                            tcp::context::state_info const* state_info)
{
    auto & session = sessions.at(session_id);

    switch (new_state)
    {
    case ENYX_TCP_STATE_ESTABLISHED:
        session.bandwidth_throttle.reset(session.configuration->send_bandwidth);
        session.state.store(Session::ESTABLISHED, std::memory_order_release);
        std::cout << "session " << session_id
            << " established" << std::endl;
        break;

    case ENYX_TCP_STATE_CLOSING:
        std::cout << "session " << session_id << " half closed" << std::endl;
        break;

    case ENYX_TCP_STATE_CLOSED:
        session.state.store(Session::UNUSED, std::memory_order_release);
        std::cout << "session " << session_id;
        if (state_info->closed.peer_reset)
            std::cout << " reset by peer";
        else
            std::cout << " closed";
        std::cout << std::endl;
        break;

    default:
        break;
    }
}

void
TcpRuntime::on_credit_change(std::uint32_t session_id,
                             std::uint32_t credit)
{
    sessions.at(session_id).tx_credit.store(credit,
                                            std::memory_order_release);
}

void
TcpRuntime::on_emi_error()
{
    std::cout << "Error on EMI stream" << std::endl;
}

void
TcpRuntime::create_monitoring_tasks(Tasks & tasks)
{
    auto task = [this]
    {
        exit_on_exception exit_on_exception{};

        event_task();
    };

    tasks.push(std::async(std::launch::async, std::move(task)));
}

void
TcpRuntime::event_task()
{
    bool no_sessions_used;
    do
    {
        no_sessions_used = true;

        // Mix status and credits fetching with check of sessions status
        for (auto i = 0UL, e = sessions.size(); i != e; ++i)
        {
            // The state is only modified in the current thread, hence
            // it can be loaded without cache synchronization
            auto state = sessions[i].state.load(std::memory_order_relaxed);
            if (state != Session::UNUSED)
                no_sessions_used = false;

            if (events_source.poll_once() < 0)
                throw std::system_error{errno, std::generic_category()};
        }
    }
    while (! is_exit_requested() && ! no_sessions_used);

    // Request exit in case loop exited because
    // all sessions are closed.
    request_exit();
}

void
TcpRuntime::open_session(std::uint32_t session_id,
                         Session & session)
{
    switch (session.configuration->type)
    {
    case SessionConfiguration::CLIENT:
        connect_session(session_id, session);
        break;

    case SessionConfiguration::SERVER:
        listen_on_session(session_id, session);
        break;

    default:
        throw std::runtime_error{"No multicast on TCP"};
    }

    session.state.store(Session::CONFIGURED, std::memory_order_release);
}

void
TcpRuntime::connect_session(std::uint32_t session_id,
                            Session & session)
{
    auto const & configuration = *session.configuration;
    ipv4_endpoint endpoint{configuration.endpoint};

    tcp::session_parameters parameters{};
    parameters.interface_id = configuration.interface_id;
    parameters.mss = configuration.mss;
    parameters.window_size = configuration.window;
    parameters.retry_timeout = configuration.retry_timeout;
    parameters.enable_instant_ack = configuration.instant_ack;

    stack.get_session(session_id).value()
         .connect(endpoint.address(), endpoint.port(), parameters).value();
}

void
TcpRuntime::listen_on_session(std::uint32_t session_id,
                              Session & session)
{
    auto const & configuration = *session.configuration;
    tcp::session_parameters parameters{};
    parameters.interface_id = configuration.interface_id;
    parameters.source_port = std::stoi(configuration.endpoint);
    parameters.mss = configuration.mss;
    parameters.window_size = configuration.window;
    parameters.retry_timeout = configuration.retry_timeout;
    parameters.enable_instant_ack = configuration.instant_ack;

    stack.get_session(session_id).value()
         .listen(parameters).value();

    if (configuration.window == 0)
        stack.get_session(session_id).value().set_window_size(0);
}

bool
TcpRuntime::is_tx_buffer_full(Session & session,
                                  std::uint32_t size)
{
    // Spin lock until enough TX credit are available
    std::uint32_t credit;
    credit = session.tx_credit.load(std::memory_order_acquire);
    if (credit < size)
    {
        ++ session.tcp_backpressure;
        return true;
    }

    // Decrease the TX credit count
    while (! session.tx_credit.compare_exchange_strong(credit,
            credit - size, std::memory_order_relaxed))
        continue;

    return false;
}

void
TcpRuntime::close_session(std::uint32_t session_id)
{
    stack.get_session(session_id).value().close().value();
}

ENYX_CORES_NAMESPACE_END
