#include "ConfigurationParser.hpp"

#include <fstream>
#include <iostream>
#include <iterator>
#include <sstream>
#include <stdexcept>
#include <utility>

#include <boost/program_options.hpp>

#include "ApplicationConfiguration.hpp"
#include "SessionConfiguration.hpp"

// Stream operators of std::chrono::milliseconds for boost options parsing
namespace std {

std::istream &
operator>>(std::istream & in, std::chrono::milliseconds & millis)
{
    std::istream::sentry sentry(in);

    if (sentry)
    {
        std::string s;
        in >> s;

        try {
            millis = std::chrono::milliseconds(std::stoul(s));
        } catch (std::invalid_argument const &) {
            throw std::invalid_argument{"invalid milliseconds value: " + s};
        } catch (std::out_of_range const &) {
            throw std::out_of_range{"milliseconds out of range: " + s};
        }
    }

    return in;
}

std::ostream &
operator<<(std::ostream & out, const std::chrono::milliseconds & millis)
{
    std::ostream::sentry sentry(out);

    if (! sentry)
        return out;

    return out << millis.count() << "ms";
}

}

ENYX_CORES_NAMESPACE_BEGIN

namespace po = boost::program_options;

namespace {

constexpr Size DEFAULT_BANDWIDTH = Size(128 * 1000 * 1000, Size::SI);

void
parse_session_configurations(ApplicationConfiguration & app_configuration,
                             std::istream & in)
{
    SessionConfiguration c{};

    po::options_description session_required{"Required arguments"};
    session_required.add_options()
        ("type,t",
            po::value<SessionConfiguration::Type>(&c.type),
            "Connect to following address\n")
        ("endpoint,e",
            po::value<std::string>(&c.endpoint),
            "Listen on following address\n")
        ("size,s",
            po::value<Size>(&c.size),
            "Amount of data to send (e.g. 8KiB, 16MiB, 1Gibit, 1GiB)\n");

    po::options_description session_optional{"Optional arguments"};
    session_optional.add_options()
        ("interface-id,i",
            po::value<std::uint32_t>(&c.interface_id)
                ->default_value(0),
            "The stack interface id to use\n")
        ("tx-bandwidth,T",
            po::value<Size>(&c.send_bandwidth)
                ->default_value(DEFAULT_BANDWIDTH),
            "Limit send bandwidth (e.g. 8kB, 16MB, 1Gbit, 1GB)\n")
        ("verify,v",
            po::value<SessionConfiguration::Verify>(&c.verify)
                ->default_value(SessionConfiguration::NONE),
            "Verify received bytes. Accepted values:\n"
            "  - none\n  - first\n  - all\n")
        ("shutdown-policy,S",
            po::value<SessionConfiguration::ShutdownPolicy>(&c.shutdown_policy)
                ->default_value(SessionConfiguration::SEND_COMPLETE),
            "Connection shutdown policy. Accepted values:\n"
            "  - send_complete\n  - receive_complete\n  - wait_for_peer\n")
        ("close-delay,C",
            po::value<std::chrono::milliseconds>(&c.close_delay)
                ->default_value(std::chrono::milliseconds::zero()),
            "Delay session close in milliseconds\n");

    po::options_description session_udp_optional{"Udp related optional arguments"};
    session_udp_optional.add_options()
        ("max-datagram-size,D",
            po::value<Range<Size>>(&c.packet_size)
                ->default_value(Range<Size>{Size{(1U << 16) - 64}}),
            "UDP and TCP packet maximum size. Accepted values:\n"
            "  - X The maximum size is equal to X\n"
            "  - X-Y The maximum size is randomly chosen for each packet "
            "between X & Y (inclusive)\n");

    po::options_description session_tcp_optional{"Tcp related optional arguments"};
    session_tcp_optional.add_options()
        ("mss,m",
            po::value<Size>(&c.mss)
                ->default_value(1460),
            "Session MSS\n")
        ("window,w",
            po::value<Size>(&c.window)
                ->default_value(4000),
            "Session window size (0 means default)\n")
        ("retry-timeout,r",
            po::value<std::uint16_t>(&c.retry_timeout)
                ->default_value(0),
            "Connection retry timeout in milliseconds (0 means default)\n")
        ("enable-instant-ack,A",
            po::bool_switch(&c.instant_ack),
            "Enable instant ack (default is disabled)\n");

    po::options_description session_all{"SESSIONS OPTIONS"};
    session_all.add(session_required)
            .add(session_optional)
            .add(session_udp_optional)
            .add(session_tcp_optional);

    // Iterate over each line of the session configuration file
    for (std::string line; std::getline(in, line); )
    {
        std::istringstream s{line};

        // Emulate a command line
        using tokenizer = std::istream_iterator<std::string>;
        std::vector<std::string> argv{"enyx-net-hw-tester"};
        argv.insert(argv.end(), tokenizer{s}, tokenizer{});

        auto session_parser = po::command_line_parser(argv)
                .options(session_all)
                .run();
        po::variables_map args;
        po::store(session_parser, args);
        po::notify(args);

        if (! args.count("type"))
            throw std::runtime_error{"--type is required"};

        if (! args.count("endpoint"))
            throw std::runtime_error{"--endpoint is required"};

        if (app_configuration.mode != ApplicationConfiguration::LOOPBACK)
        {
            if (! args.count("size") || args["size"].as<Size>() == 0)
                throw std::runtime_error{"--size is required"};
        }

        app_configuration.session_configurations.emplace_back(std::move(c));
    }
}

} // anonymous namespace

ApplicationConfiguration
parse_configuration(int argc, char ** argv)
{
    ApplicationConfiguration app_configuration{};

    std::string configuration_file;
    po::options_description cmd_optional{"Optional arguments"};
    cmd_optional.add_options()
        ("configuration-file,c",
            po::value<std::string>(&configuration_file)
                ->default_value("-"),
            "A file with one session configuration per line\n")
        ("accelerator-id,accelerator-index,a",
            po::value<std::string>(&app_configuration.accelerator)
                ->default_value("0"),
            "0-index or name of the accelerator to use\n")
        ("stack-index,s",
            po::value<std::uint32_t>(&app_configuration.stack_index)
                ->default_value(0),
            "0-index of the stack (UDP or TCP) to use\n")
        ("cpu-cores,x",
            po::value<CpuCoreIdRanges>(&app_configuration.cpu_core_ids),
            "CPU cores used to by the application threads\n")
        ("protocol,p",
            po::value<ApplicationConfiguration::Protocol>(&app_configuration.protocol)
                ->default_value(ApplicationConfiguration::TCP),
            "Protocol used to transfer data. Accepted values:\n"
            "  - tcp\n"
            "  - udp\n")
        ("mode,M",
            po::value<ApplicationConfiguration::Mode>(&app_configuration.mode)
                ->default_value(ApplicationConfiguration::BOTH),
            "Transfer mode. Accepted values:\n"
            "  - rx\n  - tx\n  - both\n  - loopback\n")
        ("enable-safe-mode,o",
            po::bool_switch(&app_configuration.safe_mode),
            "Enable safe mode (default is disabled)\n")
        ("help,h",
            "Print the command lines arguments\n");

    po::options_description cmd_all{"COMMAND LINE OPTIONS"};
    cmd_all.add(cmd_optional);

    po::variables_map args;
    auto cmd_parser = po::command_line_parser(argc, argv)
            .options(cmd_all)
            .run();
    po::store(cmd_parser, args);
    po::notify(args);

    if (args.count("help"))
    {
        std::cout << cmd_all << std::endl;
        throw std::runtime_error{"help is requested"};
    }

    if (! args.count("configuration-file"))
        throw std::runtime_error{"--configuration-file argument is required"};

    if (configuration_file == "-")
    {
        parse_session_configurations(app_configuration, std::cin);
    }
    else
    {
        std::ifstream response_file{configuration_file};
        parse_session_configurations(app_configuration, response_file);
    }

    return app_configuration;
}

ENYX_CORES_NAMESPACE_END
