#include "ApplicationConfiguration.hpp"

#include <iostream>
#include <sstream>
#include <stdexcept>
#include <utility>

ENYX_CORES_NAMESPACE_BEGIN

std::istream &
operator>>(std::istream & in, ApplicationConfiguration::Protocol & protocol)
{
    std::istream::sentry sentry(in);

    if (sentry)
    {
        std::string s;
        in >> s;

        if (s == "tcp")
            protocol = ApplicationConfiguration::TCP;
        else if (s == "udp")
            protocol = ApplicationConfiguration::UDP;
        else
            throw std::runtime_error("Unexpected protocol");
    }

    return in;
}

std::ostream &
operator<<(std::ostream & out, const ApplicationConfiguration::Protocol & protocol)
{
    std::ostream::sentry sentry(out);

    if (! sentry)
        return out;

    switch (protocol)
    {
    default:
    case ApplicationConfiguration::TCP:
        return out << "tcp";
    case ApplicationConfiguration::UDP:
        return out << "udp";
    }
}

std::istream &
operator>>(std::istream & in, ApplicationConfiguration::Mode & mode)
{
    std::istream::sentry sentry(in);

    if (sentry)
    {
        std::string s;
        in >> s;

        if (s == "tx")
            mode = ApplicationConfiguration::TX;
        else if (s == "rx")
            mode = ApplicationConfiguration::RX;
        else if (s == "both")
            mode = ApplicationConfiguration::BOTH;
        else if (s == "loopback")
            mode = ApplicationConfiguration::LOOPBACK;
        else
        {
            std::ostringstream error;
            error << "Unexpected mode (" << s << " )";
            throw std::runtime_error(error.str());
        }
    }

    return in;
}

std::ostream &
operator<<(std::ostream & out, const ApplicationConfiguration::Mode & mode)
{
    std::ostream::sentry sentry(out);

    if (! sentry)
        return out;

    switch (mode)
    {
    default:
    case ApplicationConfiguration::RX:
        return out << "rx";
    case ApplicationConfiguration::TX:
        return out << "tx";
    case ApplicationConfiguration::BOTH:
        return out << "both";
    case ApplicationConfiguration::LOOPBACK:
        return out << "loopback";
    }
}

std::ostream &
operator<<(std::ostream & out, const ApplicationConfiguration & configuration)
{
    std::ostream::sentry sentry(out);

    if (sentry)
    {
        out << "accelerator" << configuration.accelerator << "\n";
        out << "stack_index" << int(configuration.stack_index) << "\n";
        out << "mode: " << configuration.mode << "\n";
        out << "protocol: " << configuration.protocol << "\n";
        out << "safe_mode: " <<
            (configuration.safe_mode ? "enabled" : "disabled") << "\n";
        out << "cpu_core_ids" << configuration.cpu_core_ids << "\n";
        for (auto const& c : configuration.session_configurations)
            out << "---" << "\n" << c;
        out << std::flush;
    }

    return out;
}

ENYX_CORES_NAMESPACE_END
