/*
 * @file enyx-net-hw-udp-loopback-latency-test/main.cpp
 *
 * Example software code for the Enyx Development Framework product.
 * This code has been designed as a latency test for the Enyx Development
 * Framework.
 * This is a latency and coherency test for the HFP + UDP + MAC/PCS cores.
 * Packets will follow the following path :
 *
 *   ._____.    ._____.    ._________.    ._____.    ._________.
 *   |     | -> |     | -> |         | -> |     | -> |         | ---. external
 *   | SW  |    | HFP |    | Sandbox |    | UDP |    | MAC/PCS |    | loopback
 *   |_____| <- |_____| <- |_________| <- |_____| <- |_________| <--°
 *
 *
 * The test will timestamp and send a defined number of packets (See
 * "TOTAL_NUMBER_OF_PACKETS" hereafter) of a fixed size (See
 * "PAYLOAD_SIZE" hereafter) to the hardware, compute the latency of each packet
 * once received thanks to the timestamp, and at the end of the test print the
 * different percentiles of the latency achieved.
 */
#include <cstring>
#include <iostream>
#include <iomanip>

#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/a2c_stream.hpp>
#include <enyx/hw/c2a_stream.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/core_tree.hpp>
#include <enyx/hw/core.hpp>

#include <enyx/cores/data_stream/hw_source.hpp>
#include <enyx/cores/data_stream/sink.hpp>
#include <enyx/cores/data_stream/hw_sink.hpp>
#include <enyx/cores/types.hpp>
#include <enyx/cores/net_interface/net_interface.hpp>
#include <enyx/cores/udp/udp.hpp>
#include <enyx/cores/udp/data_stream.hpp>
#include <enyx/cores/hardware_ids.hpp>

#include "../lib/SignalHandler.hpp"
#include "../lib/DeviceDiscovery.hpp"
#include "Statistics.hpp"
#include "LatencyMeasurement.hpp"
#include "Cpu.hpp"
#include "ArgumentParser.hpp"

namespace hw = enyx::hw;
namespace itf = enyx::net_interface;
namespace udp = enyx::udp;
namespace data_stream = enyx::data_stream;

int main(int argc, char *argv[])
{

    enyx::install_signal_handlers();

    auto config = enyx::parse_arguments(argc, argv);

    enyx::bind_to_cpu(config.cpu_core);

    std::vector<std::uint8_t> payload(config.payload_size);
    for (std::uint16_t i = 0; i < payload.size(); i++)
        payload[i] = i & 0xff;

    enyx::hw::accelerator accelerator{enyx::create_accelerator(
                    config.accelerator)};
    enyx::hw::mmio mmio{enyx::get_first_mmio(accelerator)};
    enyx::hw::core_tree tree{hw::enumerate_cores(mmio)};
    enyx::hw::core udp_core{enyx::find_udp_core(tree, config.stack_index)};

    udp::udp stack{udp_core};
    itf::net_interfaces interfaces{udp_core};

    // Disable safe mode for latency tests
    stack.disable_safe_mode().value();

    // Create the Data Sink
    hw::c2a_stream sink_stream{enyx::filter_stream_by_name(
                                    accelerator.enumerate_c2a_streams(),
                                    config.tx_channel_name)};
    data_stream::sink data_sink{std::move(data_stream::sink_from_enyx_hw_stream(sink_stream).value())};
    data_stream::sink udp_sink{std::move(udp::create_sink(data_sink).value())};

    /* open server_session */
    auto server_session = stack.get_rx_session(0).value();
    udp::rx_session_parameters rx_params{};
    rx_params.source_port = config.port;
    rx_params.interface_id = config.server_interface_id;
    server_session.listen(rx_params).value();

    auto server_address = interfaces.get_ip_config(config.server_interface_id)
                                    .value()
                                    .get_ip();
    /* open client_session */
    auto client_session = stack.get_tx_session(0).value();
    udp::tx_session_parameters tx_params{};
    tx_params.interface_id = config.client_interface_id;
    client_session.connect(server_address,
                           rx_params.source_port,
                           tx_params).value();

    while (client_session.get_state().v() != udp::tx_session_state::OPEN
           && server_session.get_state().v() != udp::rx_session_state::LISTEN)
        continue;

    udp::sink_metadata udp_sink_metadata{};
    udp_sink_metadata.session_id = client_session.id();

    enyx::Stopwatch stopwatch;
    bool packet_received = false;
    enyx::Statistics stats{config.packet_count, config.cpu_freq};

    auto rx_cb = [&] (std::uint8_t const * data,
                      std::uint32_t size,
                      void const * metadata) {
        udp::source_metadata udp_source_metadata;
        std::memcpy(&udp_source_metadata, metadata, sizeof(udp_source_metadata));
        if (udp_source_metadata.session_id != server_session.id()) {
            std::ostringstream os{};
            os << "Received packet for unexpected session "
               << udp_source_metadata.session_id;
            throw std::runtime_error{os.str()};
        }
        stopwatch.stop();
        packet_received = true;

        // Checking packet received has the expected size
        if (size != config.payload_size)
            throw std::runtime_error("received packet size mismatch!");

        // Checking packet coherency.
        for (uint16_t i = 0; i < size; i++) {
            std::uint8_t const expected = i;
            if (data[i] != expected) {
                std::ostringstream error;

                error << std::hex
                    << "Failed coherency check (expected " << uint32_t(expected)
                    << ", got " << uint32_t(data[i]) << ")"
                    << std::endl;

                throw std::runtime_error(error.str());
            }
        }
    };

    // Create the Data Source
    hw::a2c_stream source_stream{enyx::filter_stream_by_name(
                                    accelerator.enumerate_a2c_streams(),
                                    config.rx_channel_name)};
    udp::source source{rx_cb};
    data_stream::hw_source data_source{source_stream, source};

    // For each packet
    for (std::size_t i = 0; i != config.packet_count; ++i) {
        int failure;
        packet_received = false;

        do {
            stopwatch.start();
            failure = udp_sink.send(payload.data(),
                                    payload.size(),
                                    &udp_sink_metadata);
        } while (failure != 0 && errno == EAGAIN);
        assert(failure == 0 && "enyx_cores_udp_stack_send successfully sent the"
            " packet");

        // When rx_cb is called, it sets packet_received to true, hence poll
        // until packet_received is equal to true.
        do
            data_source.poll_once();
        while (! packet_received);

        // Save the result
        stats.add_point(i, stopwatch.get());

        if (enyx::is_exit_requested())
            break;
    }

    std::cout << "exiting" << std::endl;

    server_session.close();
    client_session.close();

    if (enyx::is_exit_requested())
        std::exit(1);

    stats.sort();

    std::cout << "+------------+--------------+\n"
              << "| Percentile | Latency (ns) |\n"
              << "+------------+--------------+\n"
              << "| min        | " << std::setw(12)
              << stats.get_percentile(0) << " |\n"
              << "| 50         | " << std::setw(12) <<
              stats.get_percentile(50) << " |\n"
              << "| 75         | " << std::setw(12)
              << stats.get_percentile(75) << " |\n"
              << "| 90         | " << std::setw(12)
              << stats.get_percentile(90) << " |\n"
              << "| 95         | " << std::setw(12)
              << stats.get_percentile(95) << " |\n"
              << "| 99         | " << std::setw(12)
              << stats.get_percentile(99) << " |\n"
              << "| 99.9       | " << std::setw(12)
              << stats.get_percentile(99.9) << " |\n"
              << "| 99.99      | " << std::setw(12)
              << stats.get_percentile(99.99) << " |\n"
              << "| 99.999     | " << std::setw(12)
              << stats.get_percentile(99.999) << " |\n"
              << "| 99.9999    | " << std::setw(12)
              << stats.get_percentile(99.9999) << " |\n"
              << "| 99.99999   | " << std::setw(12)
              << stats.get_percentile(99.99999) << " |\n"
              << "| max        | " << std::setw(12)
              << stats.get_max() << " |\n"
              << "+------------+--------------+" << std::endl;
    return EXIT_SUCCESS;
}
