#include "IgmpScope.hpp"

#include <sstream>
#include <stdexcept>
#include <iomanip>

#include <enyx/cores/hardware_ids.hpp>

#include "Color.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace IgmpScope {

namespace {

namespace en = enyx;
namespace ni = en::igmp;

ni::igmp
create_igmp(hw::core & root,
           Application::StackType stack_type,
           std::uint32_t stack_index)
{
    std::vector<ni::igmp> igmp;

    for (auto & core : root)
    {
        auto const hardware_id = core.get_descriptor().hardware_id;
        if (hardware_id == enyx::hardware_ids::UDP_ULL)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::UDP||
                stack_type == Application::StackType::UDP_ULL)
                igmp.emplace_back(core);
        }
        else if (hardware_id == enyx::hardware_ids::UDP_STD)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::UDP||
                stack_type == Application::StackType::UDP_STD)
                igmp.emplace_back(core);
        }
    }

    if (igmp.size() <= stack_index)
    {
        std::ostringstream error;
        error << "Expected at least " << (stack_index + 1)
              << " stack(s)";
        throw std::runtime_error{error.str()};
    }

    return std::move(igmp[stack_index]);
}

char const USAGE[] =
"Usage: " BIN_NAME " igmp show\n"
"       " BIN_NAME " igmp ( enable | disable )\n";

void
enable(CmdLineIterator i, CmdLineIterator e, ni::igmp & igmp)
{
    ensure_end(i, e);

    igmp.enable().value();
}

void
disable(CmdLineIterator i, CmdLineIterator e, ni::igmp & igmp)
{
    ensure_end(i, e);

    igmp.disable().value();
}

void
show(CmdLineIterator i, CmdLineIterator e, ni::igmp & igmp)
{
    std::cout << "IGMP: ";
    if (igmp.enabled())
        std::cout << Color::CONFIGURED << "enabled" << Color::NONE;
    else
        std::cout << Color::UNCONFIGURED << "disabled" << Color::NONE;
    std::cout << std::endl;
}

void
parse_scope(CmdLineIterator i, CmdLineIterator e,
            ni::igmp & igmp)
{
    if (i == e || is_prefix(*i, "help"))
        std::cout << USAGE << std::flush;
    else if (is_prefix(*i, "show"))
        show(++i, e, igmp);
    else if (is_prefix(*i, "enable"))
        enable(++i, e, igmp);
    else if (is_prefix(*i, "disable"))
        disable(++i, e, igmp);
    else
    {
        std::ostringstream error;
        error <<  "Unknown ACTION '" << *i << "', try '" BIN_NAME " igmp help'";
        throw std::runtime_error{error.str()};
    }
}

} // namespace

void
run(CmdLineIterator i, CmdLineIterator e,
    hw::core & root,
    Application::StackType stack_type,
    std::uint32_t stack_index)
{
    ni::igmp igmp{create_igmp(root, stack_type, stack_index)};
    parse_scope(i, e, igmp);
}

} // namespace IgmpScope

ENYX_CORES_NAMESPACE_END
