/**
 * @file
 * This file contains the MMIO mock.
 */
#pragma once

#include <enyx/hw_c/mocking/core.h>

/**
 * This struct represents a mocked MMIO device interface
 *
 * In order to customize it, please set the context and associated callbacks.
 *
 * You may likely want to use the complete @ref enyx_hwm_mmio implementation
 * that is able to mock cores and registers.
 */
typedef struct {
    /**
     * A user defined context to be passed to all callbacks.
     */
    void *
    context;

    /**
     * Called to retrieve the size of the mmio address space.
     *
     * @param context A user defined context to be used in the callback.
     * @param size The size integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    ssize_t
    (*get_size)(void * context);

    /**
     * Called to retrieve the uid of the mmio.
     *
     * @param context A user defined context to be used in the callback.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_uid)(void * context, char * buffer, size_t capacity);

    /**
     * Called to retrieve the name of the mmio.
     *
     * @param context A user defined context to be used in the callback.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_name)(void * context, char * buffer, size_t capacity);

    /**
     * Called to write the @p value to address @p addr within
     * mmio address space.
     *
     * @param context A user defined context to be used in the callback.
     * @param addr The address of the write (in bytes).
     * @param value The integral value to write.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*write_8)(void * context, uint64_t addr, uint8_t value);

    /**
     * @copydoc write_8
     */
    int
    (*write_16)(void * context, uint64_t addr, uint16_t value);

    /**
     * @copydoc write_8
     */
    int
    (*write_32)(void * context, uint64_t addr, uint32_t value);

    /**
     * @copydoc write_8
     */
    int
    (*write_64)(void * context, uint64_t addr, uint64_t value);

    /**
     * Called to read the @p value at address @p addr within
     * mmio address space.
     *
     * @param context A user defined context to be used in the callback.
     * @param addr The address of the read (in bytes).
     * @param value The integer to fill
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*read_8)(void * context, uint64_t addr, uint8_t * value);

    /**
     * @copydoc read_8
     */
    int
    (*read_16)(void * context, uint64_t addr, uint16_t * value);

    /**
     * @copydoc read_8
     */
    int
    (*read_32)(void * context, uint64_t addr, uint32_t * value);

    /**
     * @copydoc read_8
     */
    int
    (*read_64)(void * context, uint64_t addr, uint64_t * value);
} enyx_hwm_mmio_interface;

/**
 * Create a mocked MMIO device interface
 *
 * @note This function is a helper for higher level language bindings.
 *
 * @return The new mocked MMIO interface on success or NULL on failure (with @b
 * errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hwm_mmio_interface *
enyx_hwm_mmio_interface_create();

/**
 * Destroy a mocked MMIO device interface
 *
 * @note This function is a helper for higher level language bindings.
 *
 * @param mmio The mocked MMIO interface to destroy
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hwm_mmio_interface_destroy(enyx_hwm_mmio_interface * mmio);

/**
 * This struct represents a complete mocked MMIO device with mocked cores and
 * registers.
 */
typedef struct enyx_hwm_mmio enyx_hwm_mmio;

/**
 * Create a full mocked MMIO device using a root core and the width of
 * the address space associated with this device.
 *
 * Actual address space size will be `2^addr_width` bits.
 *
 * @param mmio The mocked mmio interface to build uppon
 * @param root The root core of the tree
 * @param addr_width the width of the address space in bits
 * @return The new mocked mmio on success or NULL on failure (with @b errno set
 * accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hwm_mmio *
enyx_hwm_mmio_create(enyx_hwm_mmio_interface * mmio,
                     enyx_hwm_core * root,
                     uint8_t addr_width);

/**
 * Get the underliying mmio interface for this full mmio
 *
 * @param mmio the full mmio
 * @return the mmio interface on success or NULL on failure (with @b errno set
 * accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hwm_mmio_interface *
enyx_hwm_mmio_get_base(enyx_hwm_mmio * mmio);

/**
 * Destroy a full mmio
 *
 * @param mmio The mmio to destroy
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hwm_mmio_destroy(enyx_hwm_mmio * mmio);
