/**
 * @file
 * This file contains the a2c stream interface C mock.
 *
 * It should be used to create custom a2c_stream mocks.
 */
#pragma once

#include <stdint.h>
#include <unistd.h>

#include <enyx/hw_c/a2c_stream.h>

/**
 * Represent a mocked A2C stream interface.
 *
 * In order to implement it, please set the corresponding callbacks and context
 * consistantly.
 */
typedef struct {
    /**
     * A user defined context to be passed to all callbacks.
     */
    void *
    context;

    /**
     * Called when the stream is created.
     *
     * @param context A user defined context to be used in the callback.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*on_create)(void * context);

    /**
     * Called when the stream is destroyed.
     *
     * @param context A user defined context to be used in the callback.
     */
    void
    (*on_destroy)(void * context);

    /**
     * Called to get the uid.
     *
     * @param context A user defined context to be used in the callback.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_uid)(void * context, char * buffer, size_t capacity);


    /**
     * Called to get the name.
     *
     * @param context A user defined context to be used in the callback.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_name)(void * context, char * buffer, size_t capacity);

    /**
     * Called to get the MTU.
     *
     * @param context A user defined context to be used in the callback
     * @param mtu The mtu to fill.
     * @return 0 on success, -1 on error with @b errno set
     */
    int
    (*get_mtu)(void * context, uint32_t * mtu);

    /**
     * Called to get the size.
     *
     * @param context A user defined context to be used in the callback
     * @return the size on success, -1 on error with @b errno set
     */
    ssize_t
    (*get_size)(void * context);

    /**
     * Called to check for available data on stream. Should call @p on_data
     * handler on received packet(s).
     *
     * @note The function should return 0 if no data is available
     *
     * @param context A user defined context to be used in the callback.
     * @param on_data The handler to invoke on received packet.
     * @param opaque The opaque pointer to forward to the on_data handler.
     * @return The count of received packets.
     */
    size_t
    (*poll_once)(void * context, enyx_hw_a2c_stream_handler on_data,
                 void * opaque);

    /**
     * Called to retrieve the total packet count received by stream.
     *
     * @param context A user defined context to be used in the callback.
     * @param[out] packet_count The packet count to fill.
     * @return 0 on success, -1 on error with @b errno set
     */
    int
    (*get_packet_count)(void * context, uint32_t * packet_count);

    /**
     * Called to retrieve the stream backpressure.
     *
     * @param context A user defined context to be used in the callback.
     * @param[out] backpressure The backpressure to fill.
     * @return 0 on success, -1 on error with @b errno set
     */
    int
    (*get_backpressure)(void * context, uint32_t * backpressure);

    /**
     * Called to retrieve the total fifo errors received by stream.
     *
     * @param context A user defined context to be used in the callback.
     * @param[out] fifo_errors The fifo errors to fill.
     * @return 0 on success, -1 on error with @b errno set
     */
    int
    (*get_fifo_errors)(void * context, uint32_t * fifo_errors);

    /**
     * Called to retrieve the total truncated_count received by stream.
     *
     * @param context A user defined context to be used in the callback.
     * @param[out] truncated_count The error count to fill.
     * @return 0 on success, -1 on error with @b errno set
     */
    int
    (*get_truncated_count)(void * context, uint32_t * truncated_count);

    /**
     * Called to retrieve the total errors received by stream.
     *
     * @param context A user defined context to be used in the callback.
     * @param[out] errors The error count to fill.
     * @return 0 on success, -1 on error with @b errno set
     */
    int
    (*get_errors)(void * context, uint32_t * errors);

    /**
     * Called to retrieve the total usage received by stream.
     *
     * @param context A user defined context to be used in the callback.
     * @param[out] usage The usage to fill.
     * @return 0 on success, -1 on error with @b errno set
     */
    int
    (*get_usage)(void * context, uint32_t * usage);
} enyx_hwm_a2c_stream_interface;

/**
 * Create a mocked A2C stream interface.
 *
 * @note This function is a helper for higher level language bindings.
 *
 * @return The new mocked stream interface on success or NULL on failure (with
 * @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hwm_a2c_stream_interface *
enyx_hwm_a2c_stream_interface_create();

/**
 * Destroy a mocked A2C stream interface.
 *
 * @note This function is a helper for higher level language bindings.
 *
 * @param stream The mocked stream interface to destroy
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hwm_a2c_stream_interface_destroy(enyx_hwm_a2c_stream_interface * stream);
