/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  device mocking.
 */
#pragma once

#include <stddef.h>
#include <inttypes.h>

#include <enyx/hw_c/symbol_visibility.h>
#include <enyx/hw_c/accelerator.h>
#include <enyx/hw_c/mmio.h>
#include <enyx/hw_c/c2a_stream.h>
#include <enyx/hw_c/a2c_stream.h>

/**
 * This class contains all the function pointers
 * required in order to mock an accelerator and its
 * devices.
 *
 * @note If a function other than *_count() is not provided,
 *       the default implementation will return @b NULL
 *       or -1 and set @b errno to @b ENOSYS.
 *
 * @since 5.5.0
 */
typedef struct
{
    /**
     * Called to retrieve the count of accelerators
     *
     * @param context A user defined context to be used in the callback.
     * @return the number of detected accelerators.
     */
    size_t
    (*count_accelerators)(void * context);

    /**
     * Called to copy uid of accelerator at @p accelerator_index
     * into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_accelerator_uid)(void * context, uint32_t accelerator_index,
                           char * buffer,
                           size_t capacity);

    /**
     * Called to copy name of accelerator at @p accelerator_index
     * into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_accelerator_name)(void * context, uint32_t accelerator_index,
                            char * buffer, size_t capacity);

    /**
     * Called to retrieve the count of mmios belonging to
     * accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @return The number of mmios on success, 0 with errno set on error.
     */
    size_t
    (*count_mmios)(void * context, uint32_t accelerator_index);

    /**
     * Called to retrieve the size of the mmio address space belonging
     * to accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param size The size integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_mmio_size)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                     size_t * size);

    /**
     * Called to retrieve the uid of the mmio belonging to
     * accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_mmio_uid)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                    char * buffer, size_t capacity);

    /**
     * Called to retrieve the name of the mmio belonging to
     * accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_mmio_name)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                     char * buffer, size_t capacity);

    /**
     * Called to write the @p value to address @p addr within
     * mmio address space.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param addr The address of the write (in bytes).
     * @param value The integral value to write.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*mmio_write_8)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                    uint64_t addr, uint8_t value);

    /**
     * @copydoc mmio_write_8
     */
    int
    (*mmio_write_16)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                     uint64_t addr, uint16_t value);

    /**
     * @copydoc mmio_write_8
     */
    int
    (*mmio_write_32)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                     uint64_t addr, uint32_t value);

    /**
     * @copydoc mmio_write_8
     */
    int
    (*mmio_write_64)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                     uint64_t addr, uint64_t value);

    /**
     * Called to read the @p value at address @p addr within
     * mmio address space.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param mmio_index The mmio index.
     * @param addr The address of the read (in bytes).
     * @param value The integer to fill
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*mmio_read_8)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                   uint64_t addr, uint8_t * value);

    /**
     * @copydoc mmio_read_8
     */
    int
    (*mmio_read_16)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                    uint64_t addr, uint16_t * value);

    /**
     * @copydoc mmio_read_8
     */
    int
    (*mmio_read_32)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                    uint64_t addr, uint32_t * value);

    /**
     * @copydoc mmio_read_8
     */
    int
    (*mmio_read_64)(void * context, uint32_t accelerator_index, uint32_t mmio_index,
                    uint64_t addr, uint64_t * value);

    /**
     * Called to retrieve the number of c2a streams belonging to
     * accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @return the number of detected c2a streams accelerators.
     */
    size_t
    (*count_c2a_streams)(void * context, uint32_t accelerator_index);

    /**
     * Called to copy the uid of c2a stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_c2a_stream_uid)(void * context, uint32_t accelerator_index, uint32_t stream_index,
                          char * buffer, size_t capacity);

    /**
     * Called to copy the name of c2a stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_c2a_stream_name)(void * context, uint32_t accelerator_index, uint32_t stream_index,
                           char * buffer, size_t capacity);

    /**
     * Called to copy the mtu of c2a stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param mtu The mtu to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_c2a_stream_mtu)(void * context, uint32_t accelerator_index, uint32_t stream_index,
                          uint32_t *mtu);

    /**
     * Called to copy the size of c2a stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param size The size to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_c2a_stream_size)(void * context, uint32_t accelerator_index, uint32_t stream_index,
                           size_t *size);

    /**
     * Called to send @p size bytes of @p data over the stream @p stream_index
     * from accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param data The data to send
     * @param size The size of the data to send.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*c2a_stream_send)(void * context, uint32_t accelerator_index,
                       uint32_t stream_index,
                       void const *data,
                       uint32_t size);

    /**
     * Called to send @p vectors_count data @p vectors over the stream
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param data The data to send
     * @param size The size of the data to send.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*c2a_stream_send_io_vec)(void * context, uint32_t accelerator_index,
                              uint32_t stream_index,
                              enyx_hw_c2a_stream_io_vec const *vectors,
                              uint32_t vectors_count);

    /**
     * Called to check if the stream @p stream_index from accelerator
     * @p accelerator_index is ready.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return true if the stream is ready, false otherwise.
     */
    bool
    (*is_c2a_stream_ready)(void * context, uint32_t accelerator_index,
                           uint32_t stream_index);

    /**
     * Called to check if the stream @p stream_index from accelerator
     * @p accelerator_index is idling (i.e. it is not currently processing
     * any data).
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return true if the stream is idling, false otherwise.
     */
    bool
    (*is_c2a_stream_idle)(void * context, uint32_t accelerator_index,
                          uint32_t stream_index);

    /**
     * Called to wait until the stream @p stream_index from accelerator
     * @p accelerator_index is idling (i.e. it is not currently processing
     * any data).
     *
     * @note This function should return -1 and set @b errno if the
     *       stream is still not idling after @p us_timeout.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param us_timeout The duration of the wait.
     * @return true if the stream is idling, false otherwise.
     */
    int
    (*wait_until_c2a_stream_is_idle)(void * context, uint32_t accelerator_index,
                                     uint32_t stream_index,
                                     uint32_t us_timeout);

    /**
     * Called to retrieve the number of a2c streams belonging to
     * accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @return the number of detected a2c streams accelerators.
     */
    size_t
    (*count_a2c_streams)(void * context, uint32_t accelerator_index);

    /**
     * Called to copy the uid of a2c stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param buffer The buffer used to store the uid.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_uid)(void * context, uint32_t accelerator_index, uint32_t stream_index,
                          char * buffer, size_t capacity);

    /**
     * Called to copy the name of a2c stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param buffer The buffer used to store the name.
     * @param capacity The buffer capacity.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_name)(void * context, uint32_t accelerator_index, uint32_t stream_index,
                           char * buffer, size_t capacity);

    /**
     * Called to copy the mtu of a2c stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param mtu The mtu to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_mtu)(void * context, uint32_t accelerator_index, uint32_t stream_index,
                          uint32_t *mtu);

    /**
     * Called to copy the size of a2c stream at @p stream_index belonging
     * accelerator @p accelerator_index into @p buffer.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param size The size to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_size)(void * context, uint32_t accelerator_index, uint32_t stream_index,
                           size_t *size);

    /**
     * Called to invoke @p on_data handler on received packet(s) of stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @note The function should return 0 if no data is available
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param on_data The handler to invoke on received packet.
     * @param opaque The opaque pointer to forward to the on_data handler.
     * @return The count of received packets.
     */
    size_t
    (*poll_a2c_stream_once)(void * context, uint32_t accelerator_index, uint32_t stream_index,
                            enyx_hw_a2c_stream_handler on_data,
                            void * opaque);

    /**
     * Called to retrieve the total packet count received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param packet_count The packet count integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_packet_count)(void * context, uint32_t accelerator_index,
                                   uint32_t stream_index,
                                   uint32_t * packet_count);

    /**
     * Called to retrieve the stream at @p stream_index from
     * accelerator @p accelerator_index backpressure.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param backpressure The backpressure integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_backpressure)(void * context, uint32_t accelerator_index,
                                   uint32_t stream_index,
                                   uint32_t * backpressure);

    /**
     * Called to retrieve the total fifo errors received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param fifo_errors The fifo errors integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_fifo_errors)(void * context, uint32_t accelerator_index,
                                  uint32_t stream_index,
                                  uint32_t * fifo_errors);

    /**
     * Called to retrieve the total truncated_count received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param truncated_count The truncated_count integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_truncated_count)(void * context, uint32_t accelerator_index,
                                      uint32_t stream_index,
                                      uint32_t * truncated_count);

    /**
     * Called to retrieve the total errors received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param errors The errors integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_errors)(void * context, uint32_t accelerator_index,
                             uint32_t stream_index,
                             uint32_t * errors);

    /**
     * Called to retrieve the total usage received by stream at
     * @p stream_index from accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @param usage The usage integer to fill.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_a2c_stream_usage)(void * context, uint32_t accelerator_index,
                            uint32_t stream_index,
                            uint32_t * usage);

    /**
     * A user defined context to be passed to all callbacks.
     */
    void *
    context;

    /**
     * Called when the stream at @p stream_index belonging accelerator @p
     * accelerator_index is created.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*on_c2a_stream_create)(void * context, uint32_t accelerator_index,
                            uint32_t stream_index);

    /**
     * Called when the stream at @p stream_index belonging accelerator @p
     * accelerator_index is destroyed.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     */
    void
    (*on_c2a_stream_destroy)(void * context, uint32_t accelerator_index,
                             uint32_t stream_index);

    /**
     * Called when the stream at @p stream_index belonging accelerator @p
     * accelerator_index is created.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*on_a2c_stream_create)(void * context, uint32_t accelerator_index,
                            uint32_t stream_index);

    /**
     * Called when the stream at @p stream_index belonging accelerator @p
     * accelerator_index is destroyed.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param stream_index The stream index.
     */
    void
    (*on_a2c_stream_destroy)(void * context, uint32_t accelerator_index,
                             uint32_t stream_index);

    /**
     * Called to retrieve the @p version of the accelerator @p
     * accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     * @param version The version to set
     *
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*get_accelerator_version)(void * context, uint32_t accelerator_index,
                               enyx_hw_product_version * version);

    /**
     * Called to retrieve the user count of the accelerator
     * @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     *
     * @return The user count.
     */
    size_t
    (*get_accelerator_user_count)(void * context, uint32_t accelerator_index);

    /**
     * Called to reset the accelerator @p accelerator_index.
     *
     * @param context A user defined context to be used in the callback.
     * @param accelerator_index The accelerator index.
     *
     * @return 0 on success, -1 on error with @b errno set.
     */
    int
    (*on_accelerator_reset)(void * context, uint32_t accelerator_index);

} enyx_hw_mock;

/**
 * Create a mock object.
 *
 * @note This function is a helper for higher level language bindings.
 *
 * @return The new mock object on success or NULL on failure (with @b errno set
 * accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_mock *
enyx_hw_mock_create();

/**
 * Destroy a mock object.
 *
 * @note This function is a helper for higher level language bindings.
 *
 * @param mock The mockto destroy
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_mock_destroy(enyx_hw_mock * mock);

/**
 * Register the @p mock
 *
 * @warning Ensure there is no pending enyx_hw call when
 *          this function is used.
 *
 * @note To disable mocking, pass NULL as @p mock argument.
 *
 * @param new_mock The mock instance to set.
 * @return The previously set instance.
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_mock const*
enyx_hw_mock_set(enyx_hw_mock const* new_mock);
