/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  Accelerator MM accesses from the CPU.
 */
#pragma once

#include <stdint.h>

#include <enyx/hw_c/properties.h>
#include <enyx/hw_c/symbol_visibility.h>
#include <enyx/hw_c/fwd.h>

/**
 * This class references a mmio
 *
 * It is used to construct a mmio, but doesn't
 * constitute ownership of it.
 *
 * @since 5.0.0
 */
typedef struct enyx_hw_mmio_descriptor enyx_hw_mmio_descriptor;

/**
 * Retrieve the properties associated with a mmio @p descriptor
 *
 * @param descriptor The mmio descriptor object to query
 * @returns A pointer to the properties object
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_properties const *
enyx_hw_mmio_descriptor_get_properties(enyx_hw_mmio_descriptor const * descriptor);

/**
 * Create a MMIO from its associated @p descriptor
 *
 * @param descriptor The mmio descriptor
 * @returns The created mmio object or NULL if an error
 *          occurred (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_mmio *
enyx_hw_mmio_create(enyx_hw_mmio_descriptor const * descriptor);

/**
 * Detroy a mmio object @p c
 *
 * @param m The mmio to destroy
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_mmio_destroy(enyx_hw_mmio * m);

/**
 * Retrieve the accelerator associated with a mmio descriptor @p d
 *
 * @param d The descriptor to query
 * @returns A pointer to the accelerator object
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_accelerator *
enyx_hw_mmio_get_accelerator(enyx_hw_mmio_descriptor const * d);

/**
 * Retrieve the descriptor associated with a mmio @p m
 *
 * @param m The mmio object to query
 * @returns A pointer to the descriptor object
 *
 * @since 5.5.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_mmio_descriptor const *
enyx_hw_mmio_get_descriptor(enyx_hw_mmio const * m);

/**
 * Free a previously getted descriptor.
 *
 * @param descriptor The descriptor to free.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hw_mmio_put_descriptor(enyx_hw_mmio_descriptor const * descriptor);

/**
 * Write the @p value to address @p addr within mmio @p m address space.
 *
 * @param m The mmio device to use
 * @param addr The address of the write (in bytes).
 * @param value The integral value to write.
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_mmio_write_8(enyx_hw_mmio * m,
                     uint64_t addr,
                     uint8_t value);

/**
 * @copydoc enyx_hw_mmio_write_8
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_mmio_write_16(enyx_hw_mmio * m,
                      uint64_t addr,
                      uint16_t value);

/**
 * @copydoc enyx_hw_mmio_write_8
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_mmio_write_32(enyx_hw_mmio * m,
                      uint64_t addr,
                      uint32_t value);

/**
 * @copydoc enyx_hw_mmio_write_8
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_mmio_write_64(enyx_hw_mmio * m,
                      uint64_t addr,
                      uint64_t value);

/**
 * Read the @p value at address @p addr within mmio @p m address space.
 *
 * @param m The mmio device to use
 * @param addr The address of the read (in bytes).
 * @param value The integer to fill
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_mmio_read_8(enyx_hw_mmio * m,
                    uint64_t addr,
                    uint8_t * value);

/**
 * @copydoc enyx_hw_mmio_read_8
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_mmio_read_16(enyx_hw_mmio * m,
                     uint64_t addr,
                     uint16_t * value);

/**
 * @copydoc enyx_hw_mmio_read_8
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_mmio_read_32(enyx_hw_mmio * m,
                     uint64_t addr,
                     uint32_t * value);

/**
 * @copydoc enyx_hw_mmio_read_8
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_mmio_read_64(enyx_hw_mmio * m,
                     uint64_t addr,
                     uint64_t * value);
