/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  Core discovery.
 */
#pragma once

#include <enyx/hw_c/core.h>

#include <cerrno>
#include <cstddef>
#include <initializer_list>
#include <iostream>
#include <iterator>
#include <memory>
#include <stack>
#include <string>
#include <system_error>
#include <utility>
#include <vector>

#include <enyx/hw/namespace.hpp>
#include <enyx/hw/result.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/product.hpp>

ENYX_HW_NAMESPACE_BEGIN

/**
 * This class represents an hardware core.
 *
 * @since 5.0.0
 */
class core
{
public:
    /// The container providing the children of the core
    using children_type = std::vector<core>;

    /// The core descriptor type
    using descriptor_type = enyx_hw_core_descriptor;

    /// The core base address within mmio address space
    using base_addr_type = uint64_t;

    class iterator;

public:
    /**
     * Construct a core
     *
     * @param mmio The mmio this core belongs to
     * @param core_ptr The C API core instance to wrap
     * @param tree The tree this cores belongs to
     *
     * @since 5.0.0
     */
    core(mmio const & mmio,
         enyx_hw_core * const core_ptr,
         std::shared_ptr<enyx_hw_core_tree> const & tree) noexcept;

    /**
     * Get the number of children for this core
     *
     * @return the number of children.
     *
     * @since 5.0.0
     */
    std::size_t
    count_children() const noexcept;

    /**
     *  Retrieve the core child at @p index
     *
     *  @param index The child index
     *  @return The child
     *
     *  @since 5.7.0
     */
    core
    get_child(std::size_t index) const;

    /**
     * Retrieve the core children
     *
     * @return The children list
     *
     * @since 5.0.0
     */
    children_type
    get_children() const noexcept;

    /**
     * Retrieve the parent core
     *
     * @note This method returns ``nullptr`` when called
     *       from root core
     *
     * @return The parent core instance
     *
     * @since 5.0.0
     */
    core
    get_parent() const noexcept;

    /**
     *  Retrieve an iterator to the begin of the current subtree
     *
     *  @return The iterator
     *
     *  @since 5.7.0
     */
    iterator
    begin() const noexcept;

    /**
     *  Retrieve an iterator to the past-the-end of the current subtree
     *
     *  @note This iterator should only be compared and never dereferenced
     *
     *  @return The iterator
     *
     *  @since 5.7.0
     */
    iterator
    end() const noexcept;

    /**
     * Retrieve The root core.
     *
     * @return The root core instance.
     */
    core
    get_root() const noexcept;

    /**
     * Retrieve the core descriptor
     *
     * @return The descriptor instance
     *
     * @since 5.0.0
     */
    descriptor_type
    get_descriptor() const noexcept;

    /**
     * Get the product version of this core.
     *
     * @note The product version is *not* the core version: the core version
     * defines compatibility with the MM API while the *product* version of a
     * core is arbitrary and is specific to each type of core. On cores which
     * do not have a *product* version, this function will fail with
     * std::errc::not_supported.
     *
     * @return A result object containing the product verison on success or an
     * error on failure.
     */
    result<product_version>
    get_product_version() const noexcept;

    /**
     * Retrieve the core base address within the MMIO address space
     *
     * @return The base address as an integer
     *
     * @since 5.0.0
     */
    base_addr_type
    get_base_addr() const noexcept;

    /**
     * Retrieve the size of the core address space.
     *
     * @return The size of the address space.
     *
     * @since 5.0.0
     */
    size_t
    get_addr_space_size() const noexcept;

    /**
     * Retrieve the associated mmio
     *
     * @return The mmio
     *
     * @since 5.0.0
     */
    mmio
    get_mmio() const noexcept;

    /**
     * Advisory lock the core
     *
     * This is useful to perform dependent accesses to
     * register (i.e. paginated accesses)
     *
     * @since 5.0.0
     */
    void
    lock() noexcept;

    /**
     * Advisory try to lock the core
     *
     * This is useful to perform dependent accesses to
     * register (i.e. paginated accesses)
     *
     * @returns @b true if locked has been acquired, @b false otherwise
     *
     * @since 5.0.0
     */
    bool
    try_lock() noexcept;

    /**
     * Unlock the core
     *
     * @since 5.0.0
     */
    void
    unlock() noexcept;

    /**
     * Hardware Reset of the Core.
     *
     * @return A result object containing an error on failure.
     *
     * @since 5.0.0
     */
    result<void>
    reset() noexcept;

    /**
     * Get all descendant cores corresponding to a @p hardware_id
     *
     * @param hardware_id The hardware id we are looking for.
     *
     * @return A vector containing the corresponding cores
     *
     * @since 5.0.0
     */
    children_type
    enumerate(std::uint16_t hardware_id) const noexcept;

    /**
     * Direct access to the underlying C enyx_hw_core object.
     *
     * @return The C enyx_hw_core object.
     *
     * @since 5.0.0
     */
    enyx_hw_core *
    handle() noexcept;

    /**
     * Direct access to the underlying C enyx_hw_core object.
     *
     * @return The C enyx_hw_core object.
     *
     * @since 5.0.0
     */
    enyx_hw_core const *
    handle() const noexcept;

private:
    mmio mmio_;
    enyx_hw_core * core_;
    /* tree_ is used to keep the shared_ptr alive */
    std::shared_ptr<enyx_hw_core_tree> tree_;
};

/**
 * Represents a depth first iterator
 *
 * This iterator allows walk of the core tree:
 * @code
 *
 * auto core_tree = enyx::hw::enumerate_cores(mmio);
 * for (auto & core : core_tree)
 *     std::cout << core << std::endl;
 *
 * @endcode
 *
 * @since 5.7.0
 */
class core::iterator
{
public:
    /// @cond
    iterator(core const& root,
             std::initializer_list<std::size_t> indexes) noexcept;
    /// @endcond

    /**
     *  Pre-increment iterator
     *
     *  @return A reference to this iterator
     */
    iterator &
    operator++() noexcept;

    /**
     *  Post-increment iterator
     *
     *  @return A copy of the iterator before the increment
     */
    iterator
    operator++(int) noexcept;

    /**
     *  Compare two iterators for equality
     *
     *  @param other The other iterator to compare with
     *  @return @b true if iterators are equal, @b false otherwise
     */
    bool
    operator==(iterator const& other) const noexcept;

    /**
     *  Compare two iterators of inequality
     *
     *  @param other The other iterator to compare with
     *  @return @b true if iterators are equal, @b false otherwise
     */
    bool
    operator!=(iterator const& other) const noexcept;

    /**
     *  Dereference the iterator
     *
     *  @return A reference to the core
     */
    core &
    operator*() noexcept;

    /**
     *  Dereference the iterator
     *
     *  @return A pointer to the core
     */
    core *
    operator->() noexcept;

private:
    void
    increment() noexcept;

private:
    core current_;
    std::stack<std::size_t> indexes_;
};

/**
 * Print the core @p c into the output stream @p out
 *
 * @param out The output stream
 * @param c The core to print
 * @return A reference to @p out
 *
 * @since 5.0.0
 */
std::ostream &
operator<<(std::ostream & out, core const& c);

/**
 * Compare two cores for equality.
 *
 * @param lhs first core to compare
 * @param rhs second core to compare
 * @returns true if both core are equals.
 *
 * @since 5.0.0
 */
bool operator==(const core & lhs, const core & rhs);

/**
 * Compare a core to nullptr.
 *
 * @param lhs core to compare
 * @param rhs nullptr
 * @returns true if lhs is null.
 *
 * @since 5.0.0
 */
bool operator==(const core & lhs, const std::nullptr_t & rhs);

/**
 * Compare a core to nullptr.
 *
 * @param lhs nullptr
 * @param rhs core to compare
 * @returns true if rhs is null.
 *
 * @since 5.0.0
 */
bool operator==(const std::nullptr_t & lhs, const core & rhs);

ENYX_HW_NAMESPACE_END

/// @cond
namespace std {

template<>
struct iterator_traits<enyx::hw::core::iterator>
{
    using difference_type = std::ptrdiff_t;
    using value_type = enyx::hw::core;
    using pointer = value_type *;
    using reference = value_type &;
    using iterator_category = std::forward_iterator_tag;
};

} // namespace std
/// @endcond

#include <enyx/hw/core.ipp>
