/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  CPU to Accelerator data streaming.
 */
#pragma once

#include <chrono>

#include <enyx/hw_c/c2a_stream.h>

#include <cstdint>
#include <system_error>
#include <memory>

#include <enyx/hw/fwd.hpp>
#include <enyx/hw/namespace.hpp>
#include <enyx/hw/properties.hpp>
#include <enyx/hw/accelerator.hpp>

ENYX_HW_NAMESPACE_BEGIN

/**
 * @copydoc enyx_hw_c2a_stream_configuration
 */
using c2a_stream_configuration = enyx_hw_c2a_stream_configuration;

/**
 * @copydoc enyx_hw_c2a_stream
 */
class c2a_stream final
{
public:
    /**
     * @copydoc enyx_hw_c2a_stream_io_vec
     *
     * @since 5.0.0
     */
    using iovec = enyx_hw_c2a_stream_io_vec;

    /**
     * @copydoc enyx_hw_c2a_stream_flags
     *
     * @since 5.11.0
     */
    using flags_t = enyx_hw_c2a_stream_flags;

public:
    /**
     * Create a CPU to Accelerator channel from a @p descriptor
     * @throws std::system_error on failure.
     *
     * @since 5.0.0
     */
    explicit
    c2a_stream(c2a_stream_descriptor const& descriptor);

    /**
     * Create a CPU to Accelerator channel from a @p descriptor and
     * a @p configuration
     *
     * The @p configuration instance may be used to customize the
     * channel behavior.
     * @throws std::system_error on failure.
     *
     * @since 5.0.0
     */
    c2a_stream(c2a_stream_descriptor const& descriptor,
               c2a_stream_configuration const& configuration);

    /**
     * Deleted copy constructor
     *
     * @param old The instance not to copy from.
     *
     * @since 5.0.0
     */
    c2a_stream(c2a_stream const& old) = delete;

    /**
     * Move constructor
     */
    c2a_stream(c2a_stream &&) noexcept = default;

    /**
     * Deleted move Assignment operator
     *
     * @param old The instance to copy from
     * @return A reference to this
     *
     * @since 5.0.0
     */
    c2a_stream &
    operator=(c2a_stream const& old) = delete;

    /**
     * Assignment operator
     *
     * @note The @p old instance is left uninitialized
     *
     * @param old The instance to steal from
     * @return A reference to this
     *
     * @since 5.0.0
     */
    c2a_stream &
    operator=(c2a_stream && old) noexcept = default;

    /**
     * Send @p size byte(s) from @p data to the accelerator.
     *
     * @param data Pointer to the data to send.
     * @param size The byte(s) count to send.
     * @returns A result that contains an error on failure and may be used
     *          to throw an exception.
     *
     * @note When the accelerator can't keep up with the bandwidth, the result
     *       error will be set to @b std::errc::resource_unavailable_try_again .
     *       In that case, you may call send again.
     *
     * @since 5.0.0
     */
    result<void>
    send(void const * data, std::uint32_t size) noexcept;

    /**
     * Send @p size byte(s) from @p data to the accelerator.
     *
     * @param data Pointer to the data to send.
     * @param size The byte(s) count to send.
     * @param flags The flags to set on data packet.
     * @returns A result that contains an error on failure and may be used
     *          to throw an exception.
     *
     * @note When the accelerator can't keep up with the bandwidth, the result
     *       error will be set to @b std::errc::resource_unavailable_try_again .
     *       In that case, you may call send again.
     *
     * @since 5.11.0
     */
    result<void>
    send(void const * data, std::uint32_t size, flags_t flags) noexcept;

    /**
     * Send @p vector_count @p vectors of data to the accelerator in a row.
     *
     * @param vectors Pointer to an array of @b enyx_hfp_tx_iovec describing
     *        the data to send.
     * @param vector_count The @p vectors count.
     * @returns A result that contains an error on failure and may be used
     *          to throw an exception.
     *
     * @note When the accelerator can't keep up with the bandwidth, the result
     *       error will be set to @b std::errc::resource_unavailable_try_again .
     *       In that case, you may call send again.
     *
     * @since 5.0.0
     */
    result<void>
    send(iovec const * vectors, std::uint32_t vector_count) noexcept;

    /**
     * Send @p vector_count @p vectors of data to the accelerator in a row.
     *
     * @param vectors Pointer to an array of @b enyx_hfp_tx_iovec describing
     *        the data to send.
     * @param vector_count The @p vectors count.
     * @param flags The flags to set on data packet.
     * @returns A result that contains an error on failure and may be used
     *          to throw an exception.
     *
     * @note When the accelerator can't keep up with the bandwidth, the result
     *       error will be set to @b std::errc::resource_unavailable_try_again .
     *       In that case, you may call send again.
     *
     * @since 5.11.0
     */
    result<void>
    send(iovec const * vectors, std::uint32_t vector_count, flags_t flags)
        noexcept;

    /**
     * Check if the stream is ready to send data.
     *
     * @return true if the stream is ready to send.
     *
     * @since 5.0.0
     */
    bool
    is_ready() const noexcept;

    /**
     * Check if the stream is currently processing data
     *
     * @return true if the internal queue is empty, false otherwise.
     *
     * @since 5.0.0
     */
    bool
    is_idle() const noexcept;

    /**
     *  @brief Wait for c2a stream device to be idle.
     *  @param timeout The timeout until the function returns
     *  @returns A result object containing an error on failure.
     */
    result<void>
    wait_until_idle(std::chrono::microseconds timeout) const noexcept;

    /**
     * Retrieve the channel unique descriptor
     *
     * @return The descriptor instance
     *
     * @since 5.0.0
     */
    c2a_stream_descriptor
    get_descriptor() const;

    /**
     * Retrieve the channel MTU
     *
     * @return A result object containing either the mtu or an error.
     *
     * @since 5.6.0
     */
    result<std::uint32_t> get_mtu() const noexcept;

    /**
     * Retrieve the channel size
     *
     * @return A result object containing either the size or an error.
     *
     * @since 5.11.0
     */
    result<std::size_t> get_size() const noexcept;

    /**
     * Direct access to the underlying C c2a_stream object.
     *
     * @return The C c2a_stream object.
     *
     * @since 5.0.0
     */
    enyx_hw_c2a_stream * handle() noexcept;

    /**
     * Direct access to the underlying C c2a_stream object.
     *
     * @return The C c2a_stream object.
     *
     * @since 5.0.0
     */
    enyx_hw_c2a_stream const * handle() const noexcept;

private:
    accelerator accelerator_;
    std::unique_ptr<enyx_hw_c2a_stream,
                    decltype(&enyx_hw_c2a_stream_destroy)> c2a_stream_;
};

ENYX_HW_NAMESPACE_END

#include <enyx/hw/c2a_stream.ipp>
