/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  Accelerator to CPU data streaming.
 */
#pragma once

#include <enyx/hw_c/a2c_stream.h>

#include <cstdint>
#include <initializer_list>
#include <functional>
#include <system_error>
#include <memory>

#include <enyx/hw/fwd.hpp>
#include <enyx/hw/namespace.hpp>
#include <enyx/hw/properties.hpp>
#include <enyx/hw/accelerator.hpp>

ENYX_HW_NAMESPACE_BEGIN

/**
 * @copydoc enyx_hw_a2c_stream_configuration
 */
using a2c_stream_configuration = enyx_hw_a2c_stream_configuration;

/**
 * @copydoc enyx_hw_a2c_stream
 */
class a2c_stream final
{
public:
    /**
     * @brief A2C stream poller.
     * @tparam Handler Type of the user callback to invoke.
     *
     * This class is used to poll the A2C stream in order
     * to forward data to the handler provided in the constructor.
     *
     * @since 5.0.0
     */
    template<typename Handler>
    class poller final
    {
    public:
        /**
         * @brief Construct a poller associated with the a2c_stream @a a2c_stream_handle
         *        and the user @a handler.
         * @param a2c_stream_handle The handle to the A2C stream.
         * @param handler The user callback invoked on data.
         *
         * @since 5.0.0
         */
        poller(std::shared_ptr<enyx_hw_a2c_stream> a2c_stream_handle,
               Handler && handler);

        /**
         * @brief Poll to receive data.
         * @returns @b 1 if a message has been received
         *          (i.e. on_data invoked); otherwise @b 0.
         *
         * @since 5.0.0
         */
        std::size_t
        poll_once() noexcept;

    private:
        static void
        on_data(const std::uint8_t * data,
                std::uint32_t size,
                void * opaque);
    private:
        /*
         * shared pointer is kept for lifetime/ownership purposes but is not
         * used on the critical path for perf reasons.
         */
        std::shared_ptr<enyx_hw_a2c_stream> ptr_;
        enyx_hw_a2c_stream * handle_;
        Handler handler_;
    };
public:
    /**
     * Create an Accelerator to CPU channel from a @p descriptor
     *
     * @param descriptor The descriptor referencing the a2c_stream
     * @throws std::system_error on failure.
     *
     * @since 5.0.0
     */
    a2c_stream(a2c_stream_descriptor const& descriptor);

    /**
     * Create an Accelerator to CPU channel from a @p descriptor and
     * a @p configuration
     *
     * The @p configuration instance may be used to customize the
     * channel behavior.
     *
     * @param descriptor The descriptor referencing the a2c_stream
     * @param configuration The channel configuration
     * @throws std::system_error on failure.
     *
     * @since 5.0.0
     */
    a2c_stream(a2c_stream_descriptor const& descriptor,
               a2c_stream_configuration const& configuration);

    /**
     * Deleted copy constructor
     *
     * @param old The instance not to copy from.
     *
     * @since 5.0.0
     */
    a2c_stream(a2c_stream const& old) = delete;

    /**
     * Move constructor
     *
     * @note The @p old instance is left uninitialized
     *
     * @param old The instance to steal from
     */
    a2c_stream(a2c_stream && old) noexcept = default;

    /**
     * Deleted move Assignment operator
     *
     * @param old The instance to copy from
     * @return A reference to this
     *
     * @since 5.0.0
     */
    a2c_stream &
    operator=(a2c_stream const& old) = delete;

    /**
     * Assignment operator
     *
     * @note The @p old instance is left uninitialized
     *
     * @param old The instance to steal from
     * @return A reference to this
     *
     * @since 5.0.0
     */
    a2c_stream &
    operator=(a2c_stream && old) noexcept = default;

    /**
     * @brief Get a poller object.
     *
     * The @p handler provided must be *Copyable* or *Movable*
     * with following signature:
     *
     * @code
     * void (void const * data, std::uint32_t size)
     * @endcode
     *
     * Example:
     *
     * @code
     * auto handler = [](void const * data, std::uint32_t size) {
     *     std::cout << "received " << size << " bytes" << std::endl;
     * };
     * auto p = a2c_stream.get_poller(handler);
     * @endcode
     *
     * @tparam Handler The handler type of the argument @a handler.
     *         This type must be copyable.
     * @param handler The handler invoked on each message.
     * @returns The poller associated with both the device and
     *          the @a handler.
     * @note The @a handler is copied into the returned poller instance.
     * @see enyx::hw::a2c_stream::poller
     *
     * @since 5.0.0
     */
    template<typename Handler>
    poller<Handler>
    get_poller(Handler && handler);

    /**
     *  Retrieve the channel unique descriptor
     *
     *  @return The descriptor instance
     *
     *  @since 5.0.0
     */
    a2c_stream_descriptor
    get_descriptor() const noexcept;

    /**
     * Retrieve the channel MTU
     *
     * @return A result object containing either the mtu or an error.
     *
     * @since 5.6.0
     */
    result<std::uint32_t> get_mtu() const noexcept;

    /**
     * Retrieve the channel size
     *
     * @return A result object containing either the size or an error.
     *
     * @since 5.11.0
     */
    result<std::size_t> get_size() const noexcept;

    /**
     * Retrieve the packet_count statistic.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the packet_count or an error.
     *
     * @since 5.0.0
     */
    result<std::uint32_t> get_packet_count() const noexcept;

    /**
     * Retrieve the current number of packets that were back-pressured in the
     * accelerator due to lack of space in the accelerator to CPU buffer.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the backpressure or an error.
     *
     * @since 5.0.0
     */
    result<std::uint32_t> get_backpressure() const noexcept;

    /**
     * Retrieve the current number of packets dropped by the accelerator
     * because the accelerator to CPU buffer was full.
     *
     * @note Packets are dropped by the accelerator *only* if drop mode is
     * enabled for this device.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the fifo_errors or an error.
     *
     * @since 5.0.0
     */
    result<std::uint32_t> get_fifo_errors() const noexcept;

    /**
     * Retrieve the current number of packets dropped by the accelerator
     * because the packet was bigger than the device MTU.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the truncated_count or an error.
     *
     * @since 5.12.0
     */
    result<std::uint32_t> get_truncated_count() const noexcept;

    /**
     * Retrieve the current number of ingress packets with the error bit set
     * to 1.
     *
     * @warning Getting this statistic is a slow operation as it needs to do
     * syscalls and MM read/write operations. It should not be done in an
     * application's critical path.
     *
     * @return A result object containing either the errors or an error.
     *
     * @since 5.0.0
     */
    result<std::uint32_t> get_errors() const noexcept;

    /**
     * Retrieve the current free space in the buffer.
     *
     * @note This statistic is only available if *enable_usage* was set when
     * opening this device.
     *
     * @warning This only works with streams created with enable_usage to true.
     *
     * @return A result object containing either the current buffer usage
     * or an error.
     *
     * @since 5.0.0
     */
    result<std::uint32_t> get_usage() const noexcept;

    /**
     * Direct access to the underlying C a2c_stream object.
     *
     * @return The C a2c_stream object.
     *
     * @since 5.0.0
     */
    enyx_hw_a2c_stream * handle() noexcept;

    /**
     * Direct access to the underlying C a2c_stream object.
     *
     * @return The C a2c_stream object.
     *
     * @since 5.0.0
     */
    enyx_hw_a2c_stream const * handle() const noexcept;

private:
    accelerator accelerator_;
    std::shared_ptr<enyx_hw_a2c_stream> a2c_stream_;
};

ENYX_HW_NAMESPACE_END

#include <enyx/hw/a2c_stream.ipp>
