/**
 *  @file
 *
 *  Contains the types and functions related to the timestamping.
 */
#pragma once

#include <stdint.h>
#include <stddef.h>
#include <stdbool.h>

#include <enyx/cores_c/symbol_visibility.h>

/**
 * The format of the timestamp.
 *
 * @since 6.7.0
 */
typedef enum
{
    /// The actual format is unknown
    ENYX_PROBES_TS_UNKNOWN,

    /// This format contains 64 bits: 48=ns|16=fns
    ENYX_PROBES_TS_64_V1,

    /// This format contains 96 bits: 48=s|32=ns|16=fns
    ENYX_PROBES_TS_96_V1,
} enyx_probes_ts_format;

/**
 * Represents a duration with sub-nanosecond precision.
 *
 * @since 6.7.0
 */
typedef struct
{
    /// The sign
    bool is_neg;

    /// Elapsed s
    uint64_t s;

    /// Elapsed ns
    uint32_t ns;

    /// Fractional part
    /// e.g. 0.4 ns = 0.4 * 2^16 = 26214.4 ~= 26214 in decimal
    uint16_t fns;
} enyx_probes_duration;

/**
 * Represents a 64 bits v1 timestamp
 *
 * @since 6.7.0
 */
typedef struct
{
    /// Elapsed ns
    uint64_t ns;

    /// Fractional part
    /// e.g. 0.4 ns = 0.4 * 2^16 = 26214.4 ~= 26214 in decimal
    uint16_t fns;
} enyx_probes_ts_64_v1;

/**
 *  Compute @p a - @p b duration.
 *
 *  @param a The first timestamp.
 *  @param b The second timestamp.
 *  @return The difference duration.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_probes_duration
enyx_probes_ts_64_v1_to_duration(enyx_probes_ts_64_v1 const* a,
                                 enyx_probes_ts_64_v1 const* b);

/**
 * Parse binary timestamp.
 *
 * @param data A pointer to the received timestamp.
 * @return The parsed timestamp.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_probes_ts_64_v1
enyx_probes_ts_parse_64_v1(void const* data);

/**
 * Print @p ts into @p buffer of size @p capacity.
 *
 * @param ts A pointer to the received timestamp.
 * @param buffer The buffer to fill with the timestamp representation.
 * @param capacity The buffer size.
 * @return The count of retrieved collectors on success,
 *         -1 with @b errno set on failure.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_probes_ts_print_64_v1(enyx_probes_ts_64_v1 const* ts,
                           char * buffer, size_t capacity);
/**
 * Represents a 96 bits v1 timestamp.
 *
 * @since 6.7.0
 */
typedef struct
{
    /// Elapsed ns
    uint64_t s;

    /// Elapsed ns
    uint32_t ns;

    /// Fractional part
    /// e.g. 0.4 ns = 0.4 * 2^16 = 26214.4 ~= 26214 in decimal
    uint16_t fns;
} enyx_probes_ts_96_v1;

/**
 *  Compute @p a - @p b duration.
 *
 *  @param a The first timestamp.
 *  @param b The second timestamp.
 *  @return The difference duration.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_probes_duration
enyx_probes_ts_96_v1_to_duration(enyx_probes_ts_96_v1 const* a,
                                 enyx_probes_ts_96_v1 const* b);

/**
 * Parse binary timestamp.
 *
 * @param data A pointer to the received timestamp.
 * @return The parsed timestamp.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_probes_ts_96_v1
enyx_probes_ts_parse_96_v1(void const* data);

/**
 * Print @p ts into @p buffer of size @p capacity.
 *
 * @param ts A pointer to the received timestamp.
 * @param buffer The buffer to fill with the timestamp representation.
 * @param capacity The buffer size.
 * @return The count of retrieved collectors on success,
 *         -1 with @b errno set on failure.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_probes_ts_print_96_v1(enyx_probes_ts_96_v1 const* ts,
                           char * buffer, size_t capacity);
