#pragma once

#include <stdbool.h>
#include <stdint.h>

#include <enyx/hw_c/core.h>
#include <enyx/cores_c/symbol_visibility.h>

/**
 * Ethernet flow control subsystem object.
 *
 * Usage:
 * @code{.c}
 * int result = -1;
 * enyx_ethernet_flow_control * flow_control =
 *     enyx_ethernet_flow_control_create(udp_stack);
 *
 * if (! flow_control) {
 *     // Wrong core, ethernet flow control unavailable, or internal error.
 *     perror("enyx_ethernet_flow_control_create");
 *     goto create_failed;
 * }
 *
 * // Get the current ethernet flow control status
 * bool enabled = enyx_ethernet_flow_control_is_enabled(flow_control);
 * printf("Ethernet flow control is %s\n.", enabled ? "enabled" : "disabled");
 *
 * // Set first virtual interface (ID 0) as ethernet flow control source interface
 * // Virtual interfaces IDs are the same as in the net_interface subsystem
 * if (enyx_ethernet_flow_control_set_source_interface(flow_control, 0) < 0) {
 *     perror("enyx_ethernet_flow_control_set_source_interface");
 *     goto set_source_failed;
 * }
 *
 * // Enable ethernet flow control
 * if (enyx_ethernet_flow_control_enable(flow_control) < 0) {
 *     perror("enyx_ethernet_flow_control_enable");
 *     goto enable_failed;
 * }
 *
 * result = 0;
 *
 * enable_failed:
 * set_source_failed:
 * enyx_ethernet_flow_control_destroy(flow_control);
 * create_failed:
 * return result;
 * @endcode
 */
typedef struct enyx_ethernet_flow_control enyx_ethernet_flow_control;

/**
 * @brief Create an ethernet flow control object from a network stack.
 *
 * This module configures the following hw core:
 *
 * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
 *
 * @param stack The hardware core containing the stack.
 * @note This requires you to find a compatible enyx_hw_core core using the
 *       enyx-hw library (e.g. using enyx_hw_core_enumerate).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_ethernet_flow_control *
enyx_ethernet_flow_control_create(enyx_hw_core * stack);

/**
 * @brief Destroy an ethernet flow control object.
 *
 * @param flow_control The ethernet flow control subsystem.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_ethernet_flow_control_destroy(enyx_ethernet_flow_control * flow_control);

/**
 * @brief Check that ethernet flow control is enabled.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @return true if enabled, false otherwise.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_ethernet_flow_control_is_enabled(
        enyx_ethernet_flow_control const * flow_control);

/**
 * @brief Enable ethernet flow control.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_enable(enyx_ethernet_flow_control * flow_control);

/**
 * @brief Disable ethernet flow control.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_disable(enyx_ethernet_flow_control * flow_control);

/**
 * @brief Get the ethernet flow control buffer size.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param size The buffer size in bytes.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_get_buffer_size(
        enyx_ethernet_flow_control const * flow_control, uint32_t * size);

/**
 * @brief Get the ethernet flow control internal word size.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param size The word size in bytes.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_get_internal_word_size(
        enyx_ethernet_flow_control const * flow_control, uint32_t * size);

/**
 * @brief Get the current usage of ethernet flow control buffer.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param size The used buffer size in bytes.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_get_used_buffer(
        enyx_ethernet_flow_control const * flow_control, uint32_t * size);

/**
 * @brief Get the currently set pause threshold.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param threshold The threshold in bytes.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_get_pause_threshold(
        enyx_ethernet_flow_control const * flow_control, uint32_t * threshold);

/**
 * @brief Set the pause threshold.
 *
 * When the used buffer reach this @p threshold, ethernet flow control will
 * start sending ethernet pause frames until @b resume_threshold is reached.
 *
 * Accepted values are [internal word size;buffer size[, given value will be
 * rounded down to the internal word size.
 * Internal word size is a property configured during synthesis.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param threshold The threshold in bytes.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_set_pause_threshold(
        enyx_ethernet_flow_control * flow_control, uint32_t threshold);

/**
 * @brief Get the currently set resume threshold.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param threshold The threshold in bytes.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_get_resume_threshold(
        enyx_ethernet_flow_control const * flow_control, uint32_t * threshold);

/**
 * @brief Set the resume threshold.
 *
 * When the used buffer reach this @p threshold, ethernet flow control will
 * stop sending ethernet pause frames until @b pause_threshold is reached
 * again.
 *
 * Accepted values are [internal word size;buffer size[, given value will be
 * rounded down to the internal word size.
 * Internal word size is a property configured during synthesis.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param threshold The threshold in bytes.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_set_resume_threshold(
        enyx_ethernet_flow_control * flow_control, uint32_t threshold);

/**
 * @brief Get the virtual interface used as source of ethernet pause frames.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param interface_id The interface ID.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_get_source_interface(
        enyx_ethernet_flow_control const * flow_control, uint16_t * interface_id);

/**
 * @brief Set the virtual interface used as source of ethernet pause frames.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param interface_id The interface ID.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_set_source_interface(
        enyx_ethernet_flow_control * flow_control, uint16_t interface_id);

/**
 * @brief Get the currently set pause time used in pause frames.
 *
 * Default: maximum pause time value 65535.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param pause_time The pause time.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_get_pause_time(
        enyx_ethernet_flow_control const * flow_control, uint16_t * pause_time);

/**
 * @brief Set the pause time used in pause frames.
 *
 * This pause time will be used in ethernet pause frames sent when flow control
 * is occuring. It will inform the sender side to pause sending data for this
 * amount of time.
 *
 * The pause time is measured in units of pause "quanta", where each unit is
 * equal to (512 / net stack speed) seconds (ex. 512/10e9 for the 10G ULL).
 * See https://en.wikipedia.org/wiki/Ethernet_flow_control#Pause_frame for more
 * information.
 *
 * Warning: unless for specific cases, default value should be kept.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param pause_time The pause time.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_set_pause_time(
        enyx_ethernet_flow_control * flow_control, uint16_t pause_time);

/**
 * @brief Get the number of ethernet pause frame sent.
 *
 * @param flow_control The ethernet flow control subsystem.
 * @param count The pause frame count.
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_ethernet_flow_control_get_pause_frame_count(
        enyx_ethernet_flow_control const * flow_control, uint32_t * count);
