/// @file sink.h
#pragma once

#include <assert.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <unistd.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/data_stream/metadata.h>

/**
 *  Describe a buffer list to send
 *
 *  The @p next_buffer member is used to create
 *  a single linked list of buffers, which we be
 *  sent in order by enyx_data_stream_sink_send().
 */
struct enyx_data_stream_sink_buffer
{
    /**
     *  Required pointer to the data send
     *
     *  The data should be at least @p data_size bytes-long.
     */
    void const * data_ptr;

    /// Required byte count of data to send
    uint32_t data_size;

    /**
     *  Optional pointer to the next buffer
     *
     *  If this pointer is not @b NULL, the
     *  enyx_data_stream_sink_send_io_vec() function will
     *  insert this content at the end of the current
     *  buffer content and so one if the next buffer pointer
     *  is not @b NULL.
     */
    struct enyx_data_stream_sink_buffer const * next_buffer;
};

/// Alias shortcut
typedef struct enyx_data_stream_sink_buffer enyx_data_stream_sink_buffer;

/**
 *  This type represents an IO sink
 *
 *  The purpose of an IO sink is to send user-provided data and metadata.
 */
struct enyx_data_stream_sink
{
    /**
     *  Optional pointer to the next sink in the chain
     *
     *  This member is populated to create a sink chain, where
     *  a sink @p send_io_vec function send data to
     *  an underlying sink.
     */
    struct enyx_data_stream_sink * next_sink;

    /**
     *  Optional pointer to the sink implementation
     *
     *  This is useful if the sink needs stateful data
     *  in order to perform its task, for example the encryption
     *  key if the sink encrypts payload.
     */
    void * impl;

    /**
     *  Mandatory pointer to the sink sending function
     *
     *  This function is passed the @p next_sink value,
     *  as well as a pointer to the sink @p impl.
     *
     *  It is responsible of performing some specific processing on the data
     *  and eventually sending it (using @p next_sink and/or @p impl).
     */
    int
    (*send_io_vec)(struct enyx_data_stream_sink const* next_sink,
                            void * impl,
                            enyx_data_stream_sink_buffer const * buffer,
                            void const * metadata);

    /**
     * Mandatory pointer to the get_mtu handler, retuning the sink MTU.
     *
     * It must use the lower sink MTU to calculate its own MTU.
     */
    ssize_t
    (*get_mtu)(struct enyx_data_stream_sink const * sink);
};

/// Alias shortcut
typedef struct enyx_data_stream_sink enyx_data_stream_sink;

/**
 *  Send @p vector_count @p vectors of data to the accelerator in a row.
 *
 *  @param sink The sink instance to use
 *  @param buffer Pointer to the head of a list of buffers describing
 *         the data to send.
 *  @param metadata The message metadata chain
 *  @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 *  @note When the sink can't keep up with the bandwidth, @b errno
 *       will be set to @b EAGAIN. In that case, you may call
 *       send again.
 *
 *  Usage:
 *  @code
 *  enyx_data_stream_sink_buffer const payload = {
 *      data_ptr = "payload",
 *      data_size = 7,
 *  };
 *  enyx_data_stream_sink_buffer const header = {
 *      data_ptr = "header",
 *      data_size = 6,
 *      next_buffer = &payload,
 *
 *  int failure;
 *  do
 *      failure = enyx_data_stream_sink_send_io_vec(sink,
 *                                         buffer, sizeof(buffer),
 *                                         metadata_ptr);
 *  while (failure < 0 && errno == EGAIN);
 *
 *  if (failure < 0)
 *  {
 *      perror("enyx_data_stream_sink_send_io_vec");
 *      return -1;
 *  }
 *  @endcode
 */
static inline int
enyx_data_stream_sink_send_io_vec(enyx_data_stream_sink const* sink,
                                           enyx_data_stream_sink_buffer const * buffer,
                                           void const* metadata)
{
    assert(sink && "send_io_vec requires a sink argument");

    return sink->send_io_vec(sink->next_sink,
                                      sink->impl,
                                      buffer,
                                      metadata);
}

/**
 *  Send @p size byte(s) from @p data to the accelerator.
 *
 *  This function is a shortcut version of enyx_data_stream_sink_send_io_vec().
 *
 *  @param s The sink to use
 *  @param data Pointer to the data to send.
 *  @param size The byte(s) count to send.
 *  @param metadata The message metadata chain
 *  @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 *  @note When the sink can't keep up with the bandwidth, @b errno
 *        will be set to @b EAGAIN. In that case, you may call
 *        send again.
 *
 *  @code
 *  char buffer[] = "payload";
 *
 *  int failure;
 *  do
 *      failure = enyx_data_stream_sink_send(sink,
 *                                           buffer, sizeof(buffer),
 *                                           metadata_ptr);
 *  while (failure < 0 && errno == EGAIN);
 *
 *  if (failure < 0)
 *  {
 *      perror("enyx_data_stream_sink_send");
 *      return -1;
 *  }
 *  @endcode
 */
static inline int
enyx_data_stream_sink_send(enyx_data_stream_sink * s,
                           void const * data,
                           uint32_t size,
                           void const* metadata)
{
    enyx_data_stream_sink_buffer const buffer = {
        .data_ptr = data,
        .data_size = size,
        .next_buffer = NULL,
    };

    return enyx_data_stream_sink_send_io_vec(s, &buffer, metadata);
}

static inline ssize_t
enyx_data_stream_sink_get_mtu(enyx_data_stream_sink const * sink)
{
    return sink->get_mtu(sink);
}
