#pragma once
#include <utility>

#include <enyx/cores/types.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/cores_c/udp/session.h>

ENYX_CORES_NAMESPACE_BEGIN
namespace udp {

/// @copydoc enyx_udp_rx_session_state
enum class rx_session_state
{
    CLOSED = ENYX_UDP_RX_SESSION_STATE_CLOSED,
    LISTEN = ENYX_UDP_RX_SESSION_STATE_LISTEN,
    LISTEN_MULTICAST = ENYX_UDP_RX_SESSION_STATE_LISTEN_MULTICAST,
};

/**
 * Print a UDP RX session @p state.
 *
 * @param out The stream to use
 * @param state The state to output
 * @return A reference to the output stream
 */
std::ostream &
operator<<(std::ostream & out, rx_session_state const & state);

/// @copydoc enyx_udp_tx_session_state
enum class tx_session_state
{
    CLOSED = ENYX_UDP_TX_SESSION_STATE_CLOSED,
    OPEN = ENYX_UDP_TX_SESSION_STATE_OPEN,
};

/**
 * Print a UDP TX session @p state.
 *
 * @param out The stream to use
 * @param state The state to output
 * @return A reference to the output stream
 */
std::ostream &
operator<<(std::ostream & out, tx_session_state const & state);

/// @copydoc enyx_udp_endpoint
using endpoint = ipv4_endpoint;

/**
 * @copydoc enyx_udp_rx_session_parameters
 */
using rx_session_parameters = ::enyx_udp_rx_session_parameters;

/// @copydoc enyx_udp_rx_session_statistics
using rx_session_statistics = ::enyx_udp_rx_session_statistics;

/**
 * @copydoc enyx_udp_rx_session
 */
class rx_session
{
public:
    /**
     * Construct a session from a C session.
     *
     * @param udp The C pointer to the udp owner of the session
     * @param session The C session.
     *
     * @note This constructor should not be used directly, sessions should be
     * retrieved from @ref udp get_rx_session or get_available_rx_session
     * instead.
     */
    rx_session(std::shared_ptr<::enyx_udp> const & udp,
               ::enyx_udp_rx_session const & session);

    /**
     * Get the session id.
     *
     * @return The session id.
     */
    std::uint16_t
    id() const noexcept;

    /**
     * Listen on this session.  Port is specified via @p parameters port.
     *
     * @param parameters Session parameters
     * @return A result object containing an error on failure.
     */
    result<void>
    listen(rx_session_parameters const & parameters) noexcept;

    /**
     * Listen on @p multicast_group using this session. Port is specified via
     * @p parameters port.
     *
     * @param multicast_group multicast_group
     * @param parameters Session parameters
     * @return A result object containing an error on failure.
     */
    result<void>
    listen_multicast(ipv4_address const & multicast_group,
                     rx_session_parameters const & parameters) noexcept;

    /**
     * Close this session.
     *
     * @return A result object containing an error on failure.
     */
    result<void>
    close() noexcept;

    /**
     * Get the source port used of this session.
     *
     * @return A result object containing the source port on on success or an
     *         error on failure.
     */
    result<std::uint16_t>
    get_source_port() const noexcept;

    /**
     * @copybrief enyx_udp_rx_session_get_state
     *
     * @return A result object containing the session state or an error.
     */
    result<rx_session_state>
    get_state() const noexcept;

    /**
     * @copybrief enyx_udp_rx_session_get_statistics
     *
     * @return A result object containing the session statistics or an error.
     */
    result<rx_session_statistics>
    get_statistics() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle.
     */
    ::enyx_udp_rx_session const *
    handle() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle.
     */
    ::enyx_udp_rx_session *
    handle() noexcept;

private:
    std::shared_ptr<::enyx_udp> udp_;
    ::enyx_udp_rx_session session_;
};

/**
 * @copydoc enyx_udp_tx_session_parameters
 */
using tx_session_parameters = ::enyx_udp_tx_session_parameters;

/// @copydoc enyx_udp_tx_session_statistics
using tx_session_statistics = ::enyx_udp_tx_session_statistics;

/**
 * @copydoc enyx_udp_tx_session
 */
class tx_session
{
public:
    /**
     * Construct a session from a C session.
     *
     * @param udp The C pointer to the udp owner of the session
     * @param session The C session.
     *
     * @note This constructor should not be used directly, sessions should be
     * retrieved from @ref udp get_tx_session or get_available_tx_session
     * instead.
     */
    tx_session(std::shared_ptr<::enyx_udp> const & udp,
               ::enyx_udp_tx_session const & session);

    /**
     * Get the session id.
     *
     * @return The session id.
     */
    std::uint16_t
    id() const noexcept;

    /**
     * Connect to a server @p peer_address @p port.
     *
     * @param peer_address Remote IPv4 address to connect to (in Host Byte Order).
     * @param peer_port Remote port to connect to.
     * @param parameters Session parameters
     * @return A result object containing an error on failure.
     */
    result<void>
    connect(ipv4_address const & peer_address,
            std::uint16_t peer_port,
            tx_session_parameters const & parameters) noexcept;

    /**
     * Close this session.
     *
     * @return A result object containing an error on failure.
     *
     * @note This function does not wait for the session to be actually closed.
     *       Please check the session status afterwards.
     */
    result<void>
    close() noexcept;

    /**
     * Get the source port used of this session.
     *
     * @return A result object containing the source port on on success or an
     *         error on failure.
     */
    result<std::uint16_t>
    get_source_port() const noexcept;

    /**
     * Get the peer (remote) address and port of this session.
     *
     * @param[out] peer_address The peer IPv4 address (in Host Byte Order).
     * @param[out] peer_port The peer port.
     * @return A result object containing a pair<peer_address, peer_port> on
     *         success or an error on failure.
     */
    result<endpoint>
    get_peername() const noexcept;

    /**
     * @copybrief enyx_udp_tx_session_get_state
     *
     * @return A result object containing the session state or an error.
     */
    result<tx_session_state>
    get_state() const noexcept;

    /**
     * @copybrief enyx_udp_tx_session_get_statistics
     *
     * @return A result object containing the session statistics or an error.
     */
    result<tx_session_statistics>
    get_statistics() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle.
     */
    ::enyx_udp_tx_session const *
    handle() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle.
     */
    ::enyx_udp_tx_session *
    handle() noexcept;

private:
    std::shared_ptr<::enyx_udp> udp_;
    ::enyx_udp_tx_session session_;
};

} /* namespace udp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/udp/session.ipp>
