#pragma once

#include <cstdint>

#include <enyx/cores_c/types.h>

#include <initializer_list>
#include <iosfwd>
#include <memory>

#include <enyx/hw/core.hpp>
#include <enyx/hw/result.hpp>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

ENYX_CORES_NAMESPACE_BEGIN

/// @copydoc enyx_mac_address
class mac_address final
{
public:
    /// Mutable iterator
    using iterator = std::uint8_t *;

    /// Constant iterator
    using const_iterator = std::uint8_t const *;

public:
    /**
     * Construct an uninitialized MAC address.
     */
    mac_address() noexcept;

    /**
     * Construct a MAC from a @b C MAC object
     *
     * @param c_address The @b C MAC object to construct from
     */
    mac_address(::enyx_mac_address const& c_address) noexcept;

    /**
     * Construct a MAC from a @p str
     *
     * @param str A valid str representation of an mac address
     */
    mac_address(std::string const& str);

    /**
     * Construct a MAC from an @p integer
     *
     * @param integer A valid integer representation of a mac address
     */
    mac_address(std::uint64_t integer);

    /**
     * Construct a MAC address from an initializer list of 6 bytes
     *
     * @param l The initializer list to use
     */
    mac_address(std::initializer_list<std::uint8_t> l);

    /**
     * Construct a MAC address from a range of 6 bytes
     *
     * @tparam Iterator The iterator type
     * @param begin An iterator to the begin of the range
     * @param end An iterator to the end of the range
     */
    template<typename Iterator>
    mac_address(Iterator begin, Iterator end);

    /**
     * Return an iterator to the first byte of the MAC address
     *
     * @return An iterator to the first byte
     */
    iterator
    begin() noexcept;

    /**
     * Return an iterator to the byte following the last byte
     * of the MAC address
     *
     * @return An iterator to the byte following the last byte
     */
    iterator
    end() noexcept;

    /**
     * Return an iterator to the first byte of the MAC address
     *
     * @return An iterator to the first byte
     */
    const_iterator
    begin() const noexcept;

    /**
     * Return an iterator to the byte following the last byte
     * of the MAC address
     *
     * @return An iterator to the byte following the last byte
     */
    const_iterator
    end() const noexcept;

    /**
     * Convert a @b C++ MAC object to a @b C MAC object
     *
     * @return The @b C MAC object
     */
    operator ::enyx_mac_address * () noexcept;

    /**
     * Convert a @b C++ MAC object to a @b C MAC object
     *
     * @return The @b C MAC object
     */
    operator ::enyx_mac_address const* () const noexcept;

    /**
     * Convert this into an integer.
     *
     * @return This as an integer.
     */
    std::uint64_t
    to_int() const noexcept;

private:
    ::enyx_mac_address address_;
};

/**
 * Print a @p mac address into a stream @p out
 *
 * For example "aa:bb:cc:dd:ee:ff".
 *
 * @param out The stream to use
 * @param mac The mac address to output
 * @return A reference to the output stream
 */
std::ostream &
operator<<(std::ostream & out, mac_address const& mac);

/**
 * Fill a @p mac address from a stream @p in
 *
 * For example "aa:bb:cc:dd:ee:ff".
 *
 * @param in The stream to use
 * @param mac The mac address to fill
 * @return A reference to the input stream
 */
std::istream &
operator>>(std::istream & in, mac_address & mac);

/**
 * Compare MAC addresses @p a & @p b for equality
 *
 * @param a The first MAC address
 * @param b The second MAC address
 * @return true if both @p a & @p b are equal, false otherwise
 */
bool
operator==(mac_address const& a, mac_address const& b) noexcept;

/**
 * Compare MAC addresses @p a & @p b for difference
 *
 * @param a The first MAC address
 * @param b The second MAC address
 * @return true if both @p a & @p b are different, false otherwise
 */
bool
operator!=(mac_address const& a, mac_address const& b) noexcept;

/// @copydoc enyx_ipv4_address
class ipv4_address final
{
public:
    /// Mutable iterator
    using iterator = std::uint8_t *;

    /// Constant iterator
    using const_iterator = std::uint8_t const *;

public:
    /**
     * Construct an uninitialized IPv4 address.
     */
    ipv4_address() noexcept;

    /**
     * Construct an IPv4 from a @b C Ipv4 object
     *
     * @param c_address The @b C IPv4 object to construct from
     */
    ipv4_address(::enyx_ipv4_address const& c_address) noexcept;

    /**
     * Construct an IPv4 from a @p str
     *
     * @param str A valid str representation of an ipv4 address
     */
    ipv4_address(std::string const& str);

    /**
     * Construct an IPv4 from an @p integer
     *
     * @param integer A valid integer representation of an ipv4 address
     */
    ipv4_address(std::uint32_t integer);

    /**
     * Construct an IPv4 address from an initializer list of 4 bytes
     *
     * @param l The initializer list to use
     */
    ipv4_address(std::initializer_list<std::uint8_t> l);

    /**
     * Create an IPv4 address from a range of 4 bytes
     *
     * @tparam Iterator The iterator type
     * @param begin An iterator to the begin of the range
     * @param end An iterator to the end of the range
     */
    template<typename Iterator>
    ipv4_address(Iterator begin, Iterator end);

    /**
     * Return an iterator to the first byte of the IPv4 address
     *
     * @return An iterator to the first byte
     */
    iterator
    begin() noexcept;

    /**
     * Return an iterator to the byte following the last byte
     * of the IPv4 address
     *
     * @return An iterator to the byte following the last byte
     */
    iterator
    end() noexcept;

    /**
     * Return an iterator to the first byte of the IPv4 address
     *
     * @return An iterator to the first byte
     */
    const_iterator
    begin() const noexcept;

    /**
     * Return an iterator to the byte following the last byte
     * of the IPv4 address
     *
     * @return An iterator to the byte following the last byte
     */
    const_iterator
    end() const noexcept;

    /**
     * Convert a @b C++ IPv4 object to a @b C IPv4 object
     *
     * @return The @b C IPv4 object
     */
    operator ::enyx_ipv4_address * () noexcept;

    /**
     * Convert a @b C++ IPv4 object to a @b C IPv4 object
     *
     * @return The @b C IPv4 object
     */
    operator ::enyx_ipv4_address const* () const noexcept;

    /**
     * Convert this into an integer.
     *
     * @return This as an integer.
     */
    std::uint32_t
    to_int() const noexcept;

private:
    ::enyx_ipv4_address address_;
};

/**
 * Print a @p IPv4 address into a stream @p out
 *
 * For example "192.168.1.2".
 *
 * @param out The stream to use
 * @param ipv4 The IPv4 address to output
 * @return A reference to the output stream
 */
std::ostream &
operator<<(std::ostream & out, ipv4_address const& ipv4);

/**
 * Fill a @p IPv4 address from a stream @p in
 *
 * For example "192.168.1.2".
 *
 * @param in The stream to use
 * @param ipv4 The IPv4 address to fill
 * @return A reference to the input stream
 */
std::istream &
operator>>(std::istream & in, ipv4_address & ipv4);

/**
 * Compare IPv4 addresses @p a & @p b for equality
 *
 * @param a The first IPv4 address
 * @param b The second IPv4 address
 * @return true if both @p a & @p b are equal, false otherwise
 */
bool
operator==(ipv4_address const& a, ipv4_address const& b) noexcept;

/**
 * Compare IPv4 addresses @p a & @p b for difference
 *
 * @param a The first IPv4 address
 * @param b The second IPv4 address
 * @return true if both @p a & @p b are different, false otherwise
 */
bool
operator!=(ipv4_address const& a, ipv4_address const& b) noexcept;

/// @copydoc enyx_ipv4_endpoint
class ipv4_endpoint
{
public:
    /**
     * Construct an uninitialized IPv4 Endpoint.
     */
    ipv4_endpoint() noexcept;

    /**
     * Construct an IPv4 endpoint from a @b C IPv4 endpoint object
     *
     * @param c_endpoint The @b C object to construct from
     */
    ipv4_endpoint(::enyx_ipv4_endpoint const& c_endpoint) noexcept;

    /**
     * Construct an IPv4 endpoint from @p address and a @p port
     *
     * @param address IPv4 address
     * @param port Port of endpoint
     */
    ipv4_endpoint(ipv4_address const &address, std::uint16_t port);

    /**
     * Construct an IPv4 endpoint from a @p str
     *
     * @param str A valid str representation of an ipv4 endpoint
     */
    ipv4_endpoint(std::string const& str);

    /**
     * Access the IPv4 address of this endpoint.
     *
     * @return The IPv4 address of this endpoint.
     */
    ipv4_address address() const noexcept;

    /**
     * Access the port of this endpoint.
     *
     * @return the port of this endpoint.
     */
    std::uint16_t port() const noexcept;

    /**
     * Convert a @b C++ object to a @b C object
     *
     * @return The @b C object
     */
    operator ::enyx_ipv4_endpoint * () noexcept;

    /**
     * Convert a @b C++ object to a @b C object
     *
     * @return The @b C object
     */
    operator ::enyx_ipv4_endpoint const* () const noexcept;

private:
    ::enyx_ipv4_endpoint endpoint_;
};

/**
 * Print a @p IPv4 endpoint into a stream @p out
 *
 * For example "192.168.1.2:8080".
 *
 * @param out The stream to use
 * @param endpoint The endpoint to output
 * @return A reference to the output stream
 */
std::ostream &
operator<<(std::ostream & out, ipv4_endpoint const& endpoint);

/**
 * Fill a @p IPv4 endpoint from a stream @p in
 *
 * For example "192.168.1.2:5000".
 *
 * @param in The stream to use
 * @param endpoint The IPv4 endpoint to fill
 * @return A reference to the input stream
 */
std::istream &
operator>>(std::istream & in, ipv4_endpoint & endpoint);

/**
 * Compare endpoints @p a & @p b for equality
 *
 * @param a The first endpoint
 * @param b The second endpoint
 * @return true if both @p a & @p b are equal, false otherwise
 */
bool
operator==(ipv4_endpoint const& a, ipv4_endpoint const& b) noexcept;

/**
 * Compare endpoints @p a & @p b for difference
 *
 * @param a The first endpoint
 * @param b The second endpoint
 * @return true if both @p a & @p b are different, false otherwise
 */
bool
operator!=(ipv4_endpoint const& a, ipv4_endpoint const& b) noexcept;

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/types.ipp>
