/**
 *  @file
 *
 *  Contains the types and functions related to the
 *  Master Timestamp Generator.
 */
#pragma once

#include <enyx/cores_c/probes/mtg.h>

#include <memory>
#include <iostream>
#include <sstream>
#include <cerrno>
#include <system_error>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/probes/timestamp.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/hw/core.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_probes_mtg>
{
    void
    operator()(::enyx_probes_mtg * ptr) const
    {
        ::enyx_probes_mtg_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace probes {

/**
 * This class represents a master timestamp generator.
 *
 * @since 6.7.0
 */
class mtg final
{
/// @cond
public:
    friend std::ostream &
    operator<<(std::ostream & out, mtg const& m);
/// @endcond

public:
    /**
     * Find the first (supposedly unique) mtg of a @p subtree.
     *
     * @param root The root core.
     * @throws std::runtime_error if no mtg was found or more than one mtg was
     *         found.
     *
     * @since 6.7.0
     */
    mtg(hw::core const & root);

    /**
     * Construct a mtg from a C handle
     *
     * @param handle The mtg C handle.
     *
     * @since 6.7.0
     */
    mtg(::enyx_probes_mtg * handle);

    /**
     * Direct access to the underlying C enyx_probes_mtg object.
     *
     * @return The C enyx_probes_mtg object.
     *
     * @since 6.7.0
     */
    ::enyx_probes_mtg *
    handle() noexcept;

    /**
     * Direct access to the underlying C enyx_probes_mtg object.
     *
     * @return The C enyx_probes_mtg object.
     *
     * @since 6.7.0
     */
    ::enyx_probes_mtg const *
    handle() const noexcept;

    /**
     * Retrieve a mtg timestamp format.
     *
     * @return A result containing the timestamp format in case of success,
     * or an error otherwise.
     *
     * @since 6.7.0
     */
    result<ts_format>
    get_ts_format() const noexcept;

private:
    std::unique_ptr<::enyx_probes_mtg> mtg_;
};

/**
 * Print the MTG @p m into the output stream @p out
 *
 * @param out The output stream
 * @param m The mtg to print
 * @return A reference to @p out
 *
 * @since 6.7.0
 */
std::ostream &
operator<<(std::ostream & out, mtg const& m);

} /* namespace probes */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/probes/mtg.ipp>
