#pragma once

#include<memory>
#include<vector>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/cores_c/eth_if/eth_if.h>
#include <enyx/hw/core.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_eth_if>
{
    void
    operator()(::enyx_eth_if * ptr) const
    {
        ::enyx_eth_if_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace eth_if {

/**
 * @copybrief enyx_eth_if
 */
class eth_if {
public:
    /**
     * @copybrief enyx_eth_if_create
     *
     * @param core The eth_if HW core.
     */
    eth_if(hw::core const & core);

    /**
     * @copybrief enyx_eth_if_get_supported_data_rates
     *
     * @return A result containing a bit field of supported data rates, or an
     * error.
     */
    result<int>
    get_supported_data_rates() const noexcept;

    /**
     * @copybrief enyx_eth_if_get_i2c_support
     *
     * @return A result containing the I2C support, or an error.
     */
    result<bool>
    get_i2c_support() const noexcept;

    /**
     * @copybrief enyx_eth_if_get_i2c_controller_id
     *
     * @return A result containing the controller ID, or an error.
     */
    result<uint8_t>
    get_i2c_controller_id() const noexcept;

    /**
     * @copybrief enyx_eth_if_get_i2c_configuration
     *
     * @return A result containing the I2C configuration, or an error.
     */
    result<std::string>
    get_i2c_configuration() const noexcept;

    /**
     * @copybrief enyx_eth_if_get_data_width
     *
     * @return A result containing the data width, or an error.
     */
    result<uint32_t>
    get_data_width() const noexcept;

    /**
     * @copybrief enyx_eth_if_is_netif_enabled
     *
     * @return A result containing the host NIC activation, or an error.
     */
    result<bool>
    is_netif_enabled() const noexcept;

    /**
     * @copybrief enyx_eth_if_get_netif_id
     *
     * @return A result containing the NIC ID, or an error.
     */
    result<uint8_t>
    get_netif_id() const noexcept;

    /**
     * @copybrief enyx_eth_if_get_physical_id
     *
     * @return A result containing the physical ID, or an error.
     */
    result<uint32_t>
    get_physical_id() const noexcept;

    /**
     * @copybrief enyx_eth_if_get_physical_name
     *
     * @return A result containing the physical name, or an error.
     */
    result<std::string>
    get_physical_name() const noexcept;

    /**
     * @copybrief enyx_eth_if_get_usage_name
     *
     * @return A result containing the usage name, or an error.
     */
    result<std::string>
    get_usage_name() const noexcept;

    /**
     * @copybrief enyx_eth_if_get_raw_configuration
     *
     * @return A result containing the raw configuration, or an error.
     */
    result<std::string>
    get_raw_configuration() const noexcept;

    /**
     * Get a pointer to the underlying @b C handle @ref enyx_eth_if.
     *
     * @return A pointer to a @ref enyx_eth_if.
     */
    ::enyx_eth_if *
    handle() noexcept;

    /**
     * Get a const pointer to the underlying @b C handle @ref enyx_eth_if.
     *
     * @return A const pointer to a @ref enyx_eth_if.
     */
    ::enyx_eth_if const *
    handle() const noexcept;

private:
    enyx::hw::core eth_if_core_;
    std::unique_ptr<::enyx_eth_if> eth_if_c_;
};

} /* namespace eth_if */

ENYX_CORES_NAMESPACE_END
#include <enyx/cores/eth_if/eth_if.ipp>
