/**
 * @file
 *
 * Contains the types and functions related to the
 * cores registers.
 */
#pragma once

#include <stdint.h>

#include <enyx/hw_c/mmio.h>
#include <enyx/hw_c/core.h>
#include <enyx/hw_c/register_description.h>
#include <enyx/hw_c/symbol_visibility.h>

/**
 * Write the @p value into register @p r of core @p c.
 *
 * @param c The core to target.
 * @param r The register to write to.
 * @param value The integral value write (up to 64 bits).
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_write(enyx_hw_core const * c,
                enyx_hw_register_description const * r,
                uint64_t value);

/**
 * Read the @p value from register @p r of core @p c.
 *
 * @param c The core to target.
 * @param r The register to write to.
 * @param value The integral value write (up to 64 bits).
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_read(enyx_hw_core const * c,
               enyx_hw_register_description const * r,
               uint64_t * value);

/**
 * Read the string exposed by register couple @p index & @p value
 * located within core @p c.
 *
 * @param c The core to target.
 * @param index The register used to select.
 * @param value The integral used to read value.
 * @param buffer The buffer to fill
 * @param capacity The buffer capacity
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_read_str(enyx_hw_core const * c,
                 enyx_hw_register_description const * index,
                 enyx_hw_register_description const * value,
                 char * buffer,
                 size_t capacity);

/**
 * Poll a register @p r until the read value is equal to @p expected.
 *
 * @param c The core to target
 * @param r The register to poll
 * @param expected The expected value
 * @param timeout_ms The millisecond timeout
 * @return 0 on success, -1 on error (@b errno is set accordingly)
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_poll_until_equal_to(enyx_hw_core const * c,
                            enyx_hw_register_description const * r,
                            uint64_t expected,
                            uint64_t timeout_ms);

/**
 * Change a hardware page inside a core and wait until the page has successfuly
 * been changed.
 *
 * If @p page_ready is null, this function returns immediatly.
 *
 * @param core The core containining the page.
 * @param page_count The page_cnt register of the page.
 * @param page_index The page_id register of the page.
 * @param page_ready The page_rdy register of the page (polled to check if a
 *                   page has succesfuly been changed). May be `NULL`.
 * @param new_page_id The new page id to set.
 * @return 0 if the page has changed, -1 on failure (with @b errno set
 * accordingly).
 * @note This function blocks until the page has changed.
 *
 * @since 5.0.0
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hw_change_page(enyx_hw_core const * core,
                    enyx_hw_register_description const * page_count,
                    enyx_hw_register_description const * page_index,
                    enyx_hw_register_description const * page_ready,
                    uint64_t new_page_id);
