/**
 * @file
 *
 * Contains the types and functions related to the mocked cores.
 */
#pragma once
#include <stdint.h>
#include <enyx/hw_c/core.h>
#include <enyx/hw_c/mocking/register.h>
#include <enyx/hw_c/product.h>

/**
 * Represents a mocked core
 *
 * @since 5.9.0
 */
typedef struct enyx_hwm_core enyx_hwm_core;

/**
 * Prototype of callback used on core reset
 *
 * @since 5.11.0
 */
typedef int(* enyx_hwm_core_on_reset)(void * context);

/**
 * Create a standalone mocked core
 *
 * @note @p addr_width and @p base_addr will be updated when the created core
 * is bound to a parent.
 *
 * @param descriptor The core descriptor
 * @param addr_width The core address width
 * @param base_addr The core base address
 * @return The new core on success, NULL on error (with @b errno set
 * accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hwm_core *
enyx_hwm_core_create(enyx_hw_core_descriptor const * descriptor,
                     uint8_t addr_width,
                     uint64_t base_addr);

/**
 * Destroy the root core of a mock
 *
 * @note This function can only be called on the root core of the tree. It is a
 * no-op on the child cores.
 *
 * @param core The core to destroy
 */
ENYX_HW_C_SYMBOL_VISIBILITY void
enyx_hwm_core_destroy(enyx_hwm_core * core);

/**
 * Get the descriptor of a core
 *
 * @param core The core
 * @return The core descriptor on success, NULL on failure (with @b errno
 * set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hw_core_descriptor const *
enyx_hwm_core_get_descriptor(enyx_hwm_core const * core);

/**
 * Set the product version of a @p core.
 *
 * @param core The core to set
 * @param version The version to set
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_set_product_version(enyx_hwm_core * core,
                                  enyx_hw_product_version const * version);

/**
 * Add a register to a core
 *
 * @param core The core that will get a new register
 * @param description The description of the new register
 * @return The register child on success, NULL on failure (with @b errno set
 * accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hwm_register *
enyx_hwm_core_add_register(enyx_hwm_core * core,
                           enyx_hw_register_description const * description);

/**
 * Bind a child created with @ref enyx_hwm_core_create to a @p parent core
 *
 * @note This function should be used to simplify the integration of higher level languages
 *
 * @param parent The parent core that will get a new child
 * @param child The child to bind
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_bind_child(enyx_hwm_core * parent,
                         enyx_hwm_core * child);

/**
 * Get at most @p size children of a @p parent core
 *
 * @note if @p size is less than the number of children only @p size children
 * are obtained
 *
 * @param parent The parent core the get children from
 * @param[out] cores An array of cores to be filled with @p parent's children
 * @param size The size of @p cores
 *
 * @return The total number of children of @p parent on success, -1 on failure
 * (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hwm_core_get_children(enyx_hwm_core * parent,
                           enyx_hwm_core ** cores,
                           size_t size);

/**
 * Get at most @p size registers of a @p core
 *
 * @note If @p core has more than @p size children, only @p size children are
 * obtained
 *
 * @param core The core to get registers from
 * @param[out] regs An array of registers to be filled with @p core's registers
 * @param size The size of @p regs
 */
ENYX_HW_C_SYMBOL_VISIBILITY ssize_t
enyx_hwm_core_get_registers(enyx_hwm_core * core,
                            enyx_hwm_register ** regs,
                            size_t size);

/**
 * Write @p value at @p address of @p core. And trigger all "on_write"
 * functions of registers impacted by the write.
 *
 * @param core The core to be written
 * @param address The address of the write, relative to the core
 * @param value The value to write
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_write_8(enyx_hwm_core * core, uint64_t address, uint8_t value);

/**
 * Write @p value at @p address of @p core. And trigger all "on_write"
 * functions of registers impacted by the write.
 *
 * @param core The core to be written
 * @param address The address of the write, relative to the core
 * @param value The value to write
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_write_16(enyx_hwm_core * core, uint64_t address, uint16_t value);

/**
 * Write @p value at @p address of @p core. And trigger all "on_write"
 * functions of registers impacted by the write.
 *
 * @param core The core to be written
 * @param address The address of the write, relative to the core
 * @param value The value to write
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_write_32(enyx_hwm_core * core, uint64_t address, uint32_t value);

/**
 * Write @p value at @p address of @p core. And trigger all "on_write"
 * functions of registers impacted by the write.
 *
 * @param core The core to be written
 * @param address The address of the write, relative to the core
 * @param value The value to write
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_write_64(enyx_hwm_core * core, uint64_t address, uint64_t value);

/**
 * Read a @p value at @p address of @p core. And trigger all "on_read"
 * functions of registers impacted by the read.
 *
 * @param core The core to be read
 * @param address The address of the read, relative to the core
 * @param[out] value The value to be read
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_read_8(enyx_hwm_core * core, uint64_t address, uint8_t * value);

/**
 * Read a @p value at @p address of @p core. And trigger all "on_read"
 * functions of registers impacted by the read.
 *
 * @param core The core to be read
 * @param address The address of the read, relative to the core
 * @param[out] value The value to be read
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_read_16(enyx_hwm_core * core, uint64_t address, uint16_t * value);

/**
 * Read a @p value at @p address of @p core. And trigger all "on_read"
 * functions of registers impacted by the read.
 *
 * @param core The core to be read
 * @param address The address of the read, relative to the core
 * @param[out] value The value to be read
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_read_32(enyx_hwm_core * core, uint64_t address, uint32_t * value);

/**
 * Read a @p value at @p address of @p core. And trigger all "on_read"
 * functions of registers impacted by the read.
 *
 * @param core The core to be read
 * @param address The address of the read, relative to the core
 * @param[out] value The value to be read
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_read_64(enyx_hwm_core * core, uint64_t address, uint64_t * value);

/**
 * Update the addresses of the core tree
 *
 * @note This function is an internal helper and should not be called
 *
 * @param core The core to update
 * @param new_addr_width The new core address width
 * @param new_base_address The new base address
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_update_addresses(enyx_hwm_core * core,
                               uint8_t new_addr_width,
                               uint64_t new_base_address);

/**
 * Get the address width required to address a @p core's children
 *
 * @param core The core to get the address width
 * @return The address width
 */
ENYX_HW_C_SYMBOL_VISIBILITY uint8_t
enyx_hwm_core_get_required_width(enyx_hwm_core const * core);

/**
 * Get the base address of a @p core
 *
 * @param core The core
 * @return The core base address
 */
ENYX_HW_C_SYMBOL_VISIBILITY uint64_t
enyx_hwm_core_get_base_address(enyx_hwm_core const * core);

/**
 * Get the address space of a @p core
 *
 * @return The address space of the core, NULL on failure (with @b errno
 * set accordingly).
 */
ENYX_HW_C_SYMBOL_VISIBILITY enyx_hwm_core_address_space const *
enyx_hwm_core_get_address_space(enyx_hwm_core const * core);

/**
 * Set the reset callback of a @p core
 *
 * @param core The core to set
 * @param callback The callback to set
 * @param context A user defined object to be passed to the callback
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
ENYX_HW_C_SYMBOL_VISIBILITY int
enyx_hwm_core_set_reset_callback(enyx_hwm_core * core,
                                 enyx_hwm_core_on_reset callback,
                                 void * context);
