#pragma once

#include <string>
#include <memory>

#include <enyx/hw_c/shared_mutex.h>

#include <enyx/hw/namespace.hpp>
#include <enyx/hw/result.hpp>

/// @cond
namespace std {

template<>
struct default_delete<::enyx_hw_shared_mutex>
{
    void
    operator()(::enyx_hw_shared_mutex * ptr) const
    {
        ::enyx_hw_shared_mutex_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_HW_NAMESPACE_BEGIN

/**
 * Interprocess mutex implementation, used to protect hardware accesses between
 * multiple process invocations of enyx-hw primitives.
 */
class shared_mutex {
public:
    /**
     *  Create a shared mutex between process.
     */
    shared_mutex(std::string const & name);

    /**
     *  Try to acquire the lock.
     *
     *  @return true if the lock was acquired, false otherwise
     *
     *  @throws std::system_error on failure
     */
    bool
    try_lock() noexcept;

    /**
     *  Acquire the lock.
     *
     * @throws std::system_error on failure
     */
    void
    lock() noexcept;

    /**
     *  Release the lock.
     *
     * @throws std::system_error on failure
     */
    void
    unlock() noexcept;

    /**
     * Access to the @b C handle.
     *
     * @return a pointer to the @b C handle.
     */
    ::enyx_hw_shared_mutex const *
    handle() const noexcept;

    /**
     * Access to the @b C handle.
     *
     * @return a const pointer to the @b C handle.
     */
    ::enyx_hw_shared_mutex *
    handle() noexcept;

private:
    std::unique_ptr<::enyx_hw_shared_mutex> mutex_c_;
};

ENYX_HW_NAMESPACE_END

#include "shared_mutex.ipp"
