#pragma once

#include <cstdint>
#include <functional>
#include <memory>
#include <string>

#include <enyx/hw/register_description.hpp>
#include <enyx/hw/result.hpp>

#include <enyx/hw_c/mocking/core.h>
#include <enyx/hw_c/mocking/register.h>

ENYX_HW_NAMESPACE_BEGIN

namespace mocking {

/**
 *  Core address space storage type
 */
using core_address_space = ::enyx_hwm_core_address_space;

/**
 * This class is used to represent a mocked register
 *
 * Callback can be set to the @b on_read and @b on_write
 * attributes in order to monitor accesses.
 */
class mocked_register final
{
public:
    /**
     * Construct a register from its metadata.
     *
     * @param address_space A reference to the register's associated
     *        address space.
     * @param byte_offset Offset from the core in bytes.
     * @param bit_offset Offset from byte_offset in bits.
     * @param bit_width Size in bits.
     */
    mocked_register(core_address_space * address_space,
                    std::uint64_t byte_offset,
                    std::uint8_t bit_offset,
                    std::uint8_t bit_width);

    /**
     * Construct a register from a core and its metadata
     *
     * @note This constructor is meant to be used through @ref
     * core::add_register and not called directly.
     *
     * @param core The core to attach the register to
     * @param byte_offset Offset from the core in bytes.
     * @param bit_offset Offset from byte_offset in bits.
     * @param bit_width Size in bits.
     */
    mocked_register(::enyx_hwm_core * core,
                    std::uint64_t byte_offset,
                    std::uint8_t bit_offset,
                    std::uint8_t bit_width);

    /**
     * Construct a register from its description
     *
     * @param address_space A reference to the register's associated
     *        address space.
     * @param description The register description
     */
    mocked_register(core_address_space * address_space,
                    register_description const & description);

    /**
     * Construct a register from a core and its description
     *
     * @note This constructor is meant to be used through @ref
     * core::add_register and not called directly.
     *
     * @param core The core to attach the register to
     * @param description The register description
     */
    mocked_register(::enyx_hwm_core * core,
                    register_description const & description);

    /**
     * Deleted copy constructor
     */
    mocked_register(const mocked_register&) = delete;

    /**
     * Deleted move constructor
     */
    mocked_register &
    operator=(const mocked_register&) = delete;

    /**
     * Write the register
     *
     * This is called when the register is modified.
     *
     * This calls `on_write`.
     */
    result<void>
    trigger_write() noexcept;

    /**
     * Read the register.
     *
     * This is called everytime the register is read.
     *
     * This calls `on_read`.
     */
    result<void>
    trigger_read() noexcept;

    /**
     * Get the current value of the register
     *
     * @return the value of the register.
     */
    std::uint64_t get() const noexcept;

    /**
     * Set the value of the register
     *
     * @param value The value to set.
     */
    void set(std::uint64_t value) noexcept;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_register *
    handle() noexcept;

    /**
     * Access to the C handle
     *
     * @return the C handle
     */
    ::enyx_hwm_register const *
    handle() const noexcept;

public:
    /// Signature of the `on_read` callback
    using on_read_sig = result<std::uint64_t>();

    /**
     * A callback that is called on every read of the register.
     *
     * Default callaback calls `get()`.
     */
    std::function<on_read_sig> on_read;

    /// Signature of the `on_write` callback
    using on_write_sig = result<void>(std::uint64_t);

    /**
     * A callback that is called on every write of the register.
     *
     * Default callback calls `set()`.
     */
    std::function<on_write_sig> on_write;

    /**
     * Register description
     */
    register_description const description;

public:
    /**
     * Set the content of a string value register
     *
     * If you wish to simulate a string register, do the following:
     * @code{.cpp}
     * auto & index = mock_core.add_register(index_description);
     * auto & value = mock_core.add_register(value_description);
     *
     * std::string const fake_string = "MY_FAKE_STRING";
     * index.on_read = [&] (std::uint64_t chunk_index) {
     *     update_str(fake_string,
     *                chunk_index,
     *                index_description.bit_width / 8,
     *                value);
     *     return enyx::hw::result<void>{};
     * };
     * @endcode
     */
    static void
    update_str(std::string const& string_value,
               std::size_t chunk_index, std::size_t chunk_size,
               mocked_register & value_reg);

private:
    void _set_callbacks();

private:
    std::shared_ptr<::enyx_hwm_register> handle_;
};

} /* namespace mocking */

ENYX_HW_NAMESPACE_END

#include <enyx/hw/mocking/register.ipp>
