ENYX_HW_NAMESPACE_BEGIN

inline
core::core(mmio const & mmio,
           enyx_hw_core * const core_ptr,
           std::shared_ptr<enyx_hw_core_tree> const & tree) noexcept
    : mmio_(mmio), core_(core_ptr), tree_(tree)
{
}

inline
size_t
core::count_children() const noexcept
{
    ssize_t count = enyx_hw_core_count_children(handle());
    if (count == -1)
        return 0;
    return count;
}

inline
core
core::get_child(std::size_t index) const
{
    auto child = enyx_hw_core_get_child(handle(), index);
    if (! child)
        throw std::system_error{errno, std::generic_category()};

    return core{mmio_, child, tree_};
}

inline
core::children_type
core::get_children() const noexcept
{
    std::vector<enyx_hw_core *> ptrs(enyx_hw_core_count_children(handle()));
    /*
     * get_ptrs fails if the core used  is NULL or if ptrs is too small
     * In any case, this will mean that the returned vector will be empty,
     * which is ok.
     */
    enyx_hw_core_get_children(handle(), ptrs.data(), ptrs.size());

    children_type children;
    for (const auto &core_ptr: ptrs)
        children.emplace_back(mmio_, core_ptr, tree_);

    return children;
}

inline
core
core::get_parent() const noexcept
{
    return {mmio_, enyx_hw_core_get_parent(handle()), tree_};
}

inline
core
core::get_root() const noexcept
{
    return {mmio_, ::enyx_hw_core_tree_get_root(tree_.get()), tree_};
}

inline
core::descriptor_type
core::get_descriptor() const noexcept
{
    return *enyx_hw_core_get_descriptor(handle());
}

inline
result<product_version>
core::get_product_version() const noexcept
{
    product_version ret{};
    if (::enyx_hw_core_get_product_version(handle(), &ret) < 0)
        return std::error_code{errno, std::generic_category()};
    return ret;
}

inline
core::base_addr_type
core::get_base_addr() const noexcept
{
    base_addr_type addr;
    enyx_hw_core_get_base_address(handle(), &addr);
    return addr;
}

inline
size_t
core::get_addr_space_size() const noexcept
{
    return enyx_hw_core_get_address_space_size(handle());
}

inline
mmio
core::get_mmio() const noexcept
{
    return mmio_;
}

inline
void
core::lock() noexcept
{
    enyx_hw_core_lock(handle());
}

inline
bool
core::try_lock() noexcept
{
    return enyx_hw_core_try_lock(handle()) == 0;
}

inline
void
core::unlock() noexcept
{
    enyx_hw_core_unlock(handle());
}

inline
result<void>
core::reset() noexcept
{
    if (enyx_hw_core_reset(handle()) != 0)
        return {std::error_code{errno, std::generic_category()}};
    return {};
}

inline
enyx_hw_core *
core::handle() noexcept
{
    return core_;
}

inline
enyx_hw_core const *
core::handle() const noexcept
{
    return core_;
}

inline core::iterator
core::begin() const noexcept
{
    return iterator{*this, {0}};
}

inline core::iterator
core::end() const noexcept
{
    return iterator{get_parent(), {}};
}

inline
core::iterator::iterator(core const& current,
                         std::initializer_list<std::size_t> indexes) noexcept
    : current_(current)
    , indexes_()
{
    for (auto index : indexes)
        indexes_.push(index);
}

inline core::iterator &
core::iterator::operator++() noexcept
{
    increment();
    return *this;
}

inline core::iterator
core::iterator::operator++(int) noexcept
{
    auto const old = *this;
    increment();
    return old;
}

inline bool
core::iterator::operator==(iterator const& other) const noexcept
{
    return current_ == other.current_ && indexes_ == other.indexes_;
}

inline bool
core::iterator::operator!=(iterator const& other) const noexcept
{
    return ! (*this == other);
}

inline core &
core::iterator::operator*() noexcept
{
    return current_;
}

inline core *
core::iterator::operator->() noexcept
{
    return &current_;
}

inline void
core::iterator::increment() noexcept
{
    if (indexes_.empty())
        return;

    if (indexes_.top() < current_.count_children()) {
        current_ = current_.get_child(indexes_.top()++);
        indexes_.push(0);
    }
    else {
        current_ = current_.get_parent();
        indexes_.pop();
        increment();
    }
}

inline
std::ostream &
operator<<(std::ostream & out, core const & core)
{
    auto descriptor = core.get_descriptor();
    out << descriptor.hardware_id
        << "\t" << (uint16_t)descriptor.major
        << "." << (uint16_t)descriptor.minor
        << "." << descriptor.revision
        << "\t" << core.get_base_addr()
        << "\t" << core.get_addr_space_size();
    return out;
}

inline
bool
operator==(const core & lhs, const core & rhs)
{
    return lhs.handle() == rhs.handle();
}

inline
bool
operator==(const core & lhs, const std::nullptr_t & rhs)
{
    return lhs.handle() == rhs;
}

inline
bool
operator==(const std::nullptr_t & lhs, const core & rhs)
{
    return rhs == lhs;
}

inline
std::vector<core>
core::enumerate(std::uint16_t hardware_id) const noexcept
{
    std::vector<enyx_hw_core *> cores(100);
    auto total = enyx_hw_core_enumerate(handle(), hardware_id,
                                        cores.data(), cores.size());
    if (total == -1)
        return {};

    std::vector<core> ret;
    for (size_t i = 0; i != (size_t)total; ++i)
        ret.emplace_back(mmio_, cores[i], tree_);
    return ret;
}

ENYX_HW_NAMESPACE_END
