/*
 * HFP tx implementation.
 * Copyright (C) 2016 David Keller <david.keller@enyx.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#include "enyx_hfp_tx_usr.h"

#include <linux/version.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/dma-mapping.h>
#include <linux/module.h>
#include <linux/interrupt.h>

#include "enyx_hfp_common.h"
#include "enyx_hfp_tx_dev.h"

/* A 8-bit r/w 8 register: reset */
#define HFP_TX_RESET                        HFP_REG_ADDR(4, 0)

/* A 16-bit r/o register: cache line shift min */
#define HFP_TX_CACHE_LINE_SHIFT_MIN         HFP_REG_ADDR(5, 0)

/* A 16-bit r/o register: cache line shift max */
#define HFP_TX_CACHE_LINE_SHIFT_MAX         HFP_REG_ADDR(5, 2)

/* A 8-bit r/o register: channel count */
#define HFP_TX_DEVICE_COUNT                 HFP_REG_ADDR(6, 0)

/* A 16-bit r/o register: channel mtu */
#define HFP_TX_MTU                          HFP_REG_ADDR(6, 2)

/* A 16-bit r/o register: per device buffer size shit */
#define HFP_TX_BUFFER_SIZE_SHIFT            HFP_REG_ADDR(7, 0)

/* A 8-bit r/w register: cpu cache line shift */
#define HFP_TX_CACHE_LINE_SHIFT             HFP_REG_ADDR(8, 0)

/* A 8-bit r/w register: cmd exec */
#define HFP_TX_CMD_EXEC                     HFP_REG_ADDR(9, 0)

/* A 16-bit r/w register: cmd device id */
#define HFP_TX_CMD_DEVICE_ID                HFP_REG_ADDR(9, 2)

/* A 64-bit w/o register: cmd ctrl phys addr */
#define HFP_TX_CMD_CTRL_ADDR                HFP_REG_ADDR(10, 0)

/* A 32-bit r/o register: buffer offset from bar0 (dword) */
#define HFP_TX_BUFFER_BASE_OFFSET_DW        HFP_REG_ADDR(12, 0)

/* A 32-bit r/o register: buffer increment */
#define HFP_TX_BUFFER_INCREMENT             HFP_REG_ADDR(13, 0)

/* A 8-bit w/o register: info device id */
#define HFP_TX_INFO_DEVICE_ID               HFP_REG_ADDR(50, 0)

/* A 8-bit w/o register: info usage str index */
#define HFP_TX_INFO_USAGE_STR_INDEX         HFP_REG_ADDR(50, 2)

/* A 8-bit r/o register: info device */
#define HFP_TX_INFO_PAGE_READY              HFP_REG_ADDR(51, 0)

/* A 32-bit r/o register: usage str */
#define HFP_TX_INFO_USAGE_STR               HFP_REG_ADDR(52, 0)

/* A 32-bit r/o register: device buffer size */
#define HFP_TX_DEVICE_BUFFER_SIZE           HFP_REG_ADDR(53, 0)

/* A 16 bit r/o register: Current channel MTU size */
#define HFP_TX_DEVICE_MTU                   HFP_REG_ADDR(54, 0)


struct enyx_hfp_tx_channel {
    uint8_t __iomem * buffer_start;
    uint32_t buffer_offset;
    uint32_t buffer_size;

    volatile uint32_t * buffer_rd_ptr;
    uint32_t buffer_free_size;
    uint16_t mtu;
};

union enyx_hfp_tx_msg_hdr
{
    struct
    {
        uint16_t data_size;
    }s;
    uint64_t b[4];
};

struct enyx_hfp_tx_impl {
    struct enyx_hfp_device * parent;

    uint16_t channel_count;
    uint16_t global_mtu;

    /* Ctrl */
    size_t ctrl_size;
    size_t ctrl_increment;
    void * ctrl_virt_addr;
    dma_addr_t ctrl_bus_addr;
    struct device * dma;

    /* Buffer */
    void __iomem * buffer_io_addr;
    size_t buffer_increment;
    size_t buffer_size;
    phys_addr_t buffer_phys_addr;

    size_t coherency_line_size;

    struct enyx_hfp_tx_dev methods;

    struct enyx_hfp_tx_channel channels[];
};

#define dev_to_enyx_hfp_tx_impl(n) ({ \
    struct enyx_hfp_tx_dev * __c = n; \
    container_of(__c, struct enyx_hfp_tx_impl, methods);})

static int
uninstall_ctrl_dma_on_channel(struct enyx_hfp_tx_impl * impl,
                              uint16_t id)
{
    int err = 0;

    down(&impl->parent->lock);

    if (enyx_hfp_read8(impl->parent, HFP_TX_CMD_EXEC)) {
        dev_err(&impl->parent->device,
                "Failed to uninstall dma ctrl on "
                "channel %u (cmd pending)\n", id);
        err = -EBUSY;
        goto cmd_failed;
    }

    enyx_hfp_write16(impl->parent, HFP_TX_CMD_DEVICE_ID, id);
    enyx_hfp_write64(impl->parent, HFP_TX_CMD_CTRL_ADDR, 0);

    enyx_hfp_read8(impl->parent, HFP_TX_CMD_EXEC);
    enyx_hfp_write8(impl->parent, HFP_TX_CMD_EXEC, 1);

    if (enyx_hfp_wait_until_val_is(impl->parent, HFP_TX_CMD_EXEC, 0) < 0) {
        dev_err(&impl->parent->device,
                "Failed to uninstall dma ctrl on channel %u\n", id);
        err = -EIO;
        goto cmd_failed;
    }

    dev_dbg(&impl->parent->device, "Uninstalled channel %d dma ctrl\n", id);

cmd_failed:
    up(&impl->parent->lock);

    return err;
}

static int
install_ctrl_dma_on_channel(struct enyx_hfp_tx_impl * impl, uint16_t id)
{
    const dma_addr_t base_addr = impl->ctrl_bus_addr;
    const uint64_t ctrl_phys_addr = base_addr + id * impl->ctrl_increment;
    int err = 0;

    if (down_interruptible(&impl->parent->lock))
        return -ERESTARTSYS;

    /* Reset first header of each ctrl */
    memset(impl->ctrl_virt_addr + id * impl->ctrl_increment,
           0, sizeof(uint64_t));

    if (enyx_hfp_read8(impl->parent, HFP_TX_CMD_EXEC)) {
        dev_err(&impl->parent->device,
                "Failed to install dma ctrl on "
                "channel %u (cmd pending)\n", id);
        err = -EBUSY;
        goto cmd_failed;
    }

    enyx_hfp_write16(impl->parent, HFP_TX_CMD_DEVICE_ID, id);
    enyx_hfp_write64(impl->parent, HFP_TX_CMD_CTRL_ADDR, ctrl_phys_addr);

    enyx_hfp_read8(impl->parent, HFP_TX_CMD_EXEC);
    enyx_hfp_write8(impl->parent, HFP_TX_CMD_EXEC, 1);

    if (enyx_hfp_wait_until_val_is(impl->parent, HFP_TX_CMD_EXEC, 0) < 0) {
        dev_err(&impl->parent->device,
                "Failed to install dma ctrl on "
                "channel %u (cmd failed)\n", id);
        err = -EIO;
        goto cmd_failed;
    }

    dev_dbg(&impl->parent->device,
            "Installed channel %d dma ctrl at %llx\n", id, ctrl_phys_addr);

cmd_failed:
    up(&impl->parent->lock);
    return err;
}

static size_t
get_info_16_v0_2(struct enyx_hfp_tx_impl * impl,
                 uint16_t channel_id,
                 int register_offset)
{
    return enyx_hfp_read16_locked_v0_2(impl->parent,
            register_offset,
            HFP_TX_INFO_DEVICE_ID,
            HFP_TX_INFO_PAGE_READY,
            channel_id);
}

static int
enable_channel_v0_0(struct enyx_hfp_tx_dev * dev,
                    uint16_t id)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    struct enyx_hfp_tx_channel * channel = &impl->channels[id];
    int err;

    if (id >= impl->channel_count) {
        err = -EINVAL;
        goto channel_id_check_failed;
    }

    channel->buffer_start = impl->buffer_io_addr + id * impl->buffer_increment;
    channel->buffer_offset = 0;
    channel->buffer_rd_ptr = impl->ctrl_virt_addr + id * impl->ctrl_increment;

    err = install_ctrl_dma_on_channel(impl, id);

channel_id_check_failed:
    return err;
}

static int
enable_all_channels_v0_0(struct enyx_hfp_tx_dev * dev)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    uint16_t i, channel_count = impl->channel_count;

    for (i = 0; i != channel_count; ++i)
        if (dev->enable_channel(dev, i) < 0)
            goto enyx_hfp_tx_dev_enable_channel_failed;

    return 0;

enyx_hfp_tx_dev_enable_channel_failed:
    while (i)
        dev->disable_channel(dev, --i);
    return -EIO;
}

static void
disable_channel_v0_0(struct enyx_hfp_tx_dev * dev,
                     uint16_t channel_id)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    if (channel_id >= impl->channel_count)
        return;

    uninstall_ctrl_dma_on_channel(impl, channel_id);
}

static void
disable_all_channels_v0_0(struct enyx_hfp_tx_dev * dev)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    uint16_t i = impl->channel_count;

    while (i)
        dev->disable_channel(dev, --i);
}

static int
mmap_ctrl_v0_0(struct enyx_hfp_tx_dev * dev, struct vm_area_struct *vma,
               uint16_t channel_id)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);

    size_t size = impl->ctrl_increment;
    void * virt_addr = impl->ctrl_virt_addr + channel_id * size;
    unsigned long vm_size = vma->vm_end - vma->vm_start,
                  vm_offset = vma->vm_pgoff << PAGE_SHIFT;
    int err;

    dev_dbg(&impl->parent->device,
            "Requested ctrl mapping of %lu bytes\n", impl->ctrl_size);

    if (vm_size > size - vm_offset) {
        dev_err(&impl->parent->device,
                "Could not grant ctrl mapping of %lu bytes as it is bigger than ctrl increment (%lu) - offset (%lu)\n",
                vm_size, size, vm_offset);

        err = -EINVAL;
        goto err_invalid_arg;
    }

    err = remap_pfn_range(vma,
                          vma->vm_start,
                          virt_to_phys(virt_addr + vm_offset) >> PAGE_SHIFT,
                          vm_size,
                          vma->vm_page_prot);

err_invalid_arg:
    return err;
}

static int
mmap_buffer_v0_0(struct enyx_hfp_tx_dev * dev, struct vm_area_struct *vma,
                 uint16_t channel_id)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    unsigned long io_offset = vma->vm_pgoff << PAGE_SHIFT,
                  io_start = impl->buffer_phys_addr + io_offset +
                             channel_id * impl->buffer_increment,
                  vm_size = vma->vm_end - vma->vm_start;
    int err;

    dev_dbg(&impl->parent->device,
            "Requested buffer mapping of %lu bytes at %lx\n",
            vm_size, io_start);

    if (vm_size > PAGE_ALIGN(impl->buffer_increment - io_offset)) {
        dev_err(&impl->parent->device,
                "Could not grant buffer mapping of %lu bytes as it is bigger than buffer increment (%lu) - offset (%lu)\n",
                vm_size, impl->buffer_increment, io_offset);
        err = -EINVAL;
        goto err_invalid_arg;
    }

    err = io_remap_pfn_range(vma,
                             vma->vm_start,
                             io_start >> PAGE_SHIFT,
                             vm_size,
                             pgprot_writecombine(vma->vm_page_prot));

err_invalid_arg:
    return err;
}

void
enyx_hfp_tx_dev_destroy(struct enyx_hfp_tx_dev * dev)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    dma_free_coherent(impl->dma,
                      impl->ctrl_size,
                      impl->ctrl_virt_addr,
                      impl->ctrl_bus_addr);
    iounmap(impl->buffer_io_addr);

    put_device(&impl->parent->device);

    kfree(impl);
}
EXPORT_SYMBOL(enyx_hfp_tx_dev_destroy);

static int
setup_cache_line(struct enyx_hfp_tx_impl * impl,
                 size_t coherency_line_size)
{
    const size_t min_size = 1ULL << enyx_hfp_read16(impl->parent,
            HFP_TX_CACHE_LINE_SHIFT_MIN);

    const size_t max_size = 1ULL << enyx_hfp_read16(impl->parent,
            HFP_TX_CACHE_LINE_SHIFT_MAX);

    dev_dbg(&impl->parent->device, "Reported min CL size is %zu\n", min_size);
    dev_dbg(&impl->parent->device, "Reported max CL size is %zu\n", max_size);

    coherency_line_size = max(coherency_line_size, min_size);

    if (max_size < coherency_line_size) {
        dev_err(&impl->parent->device,
                "Reported max CL size (%zu) is smaller than CPU CL size (%zu)\n",
                max_size, coherency_line_size);
        goto cache_size_check_failed;
    }

    impl->coherency_line_size = coherency_line_size;

    enyx_hfp_write8(impl->parent, HFP_TX_CACHE_LINE_SHIFT,
               fls(coherency_line_size) - 1);

    return 0;

cache_size_check_failed:
    return -EIO;
}

static int
alloc_dma_ctrl(struct enyx_hfp_tx_impl * impl)
{
    impl->dma = &impl->parent->device;

    while (impl->dma && ! impl->dma->dma_mask)
            impl->dma = impl->dma->parent;

    if (! impl->dma) {
        dev_err(&impl->parent->device,
                "Can't find a DMA capable device in this device hierarchy\n");
        goto is_device_dma_capable_failed;
    }

    impl->ctrl_increment = PAGE_SIZE;
    impl->ctrl_size = impl->ctrl_increment * impl->channel_count;

    impl->ctrl_virt_addr = dma_alloc_coherent(impl->dma,
                                              impl->ctrl_size,
                                              &impl->ctrl_bus_addr,
                                              GFP_KERNEL);
    if (! impl->ctrl_virt_addr) {
        dev_err(&impl->parent->device,
                "Can't allocate %zu bytes DMA ctrl\n", impl->ctrl_size);
        goto dma_alloc_attrs_failed;
    }

    dev_dbg(impl->dma, "Allocated a %zu bytes DMA ctrl\n", impl->ctrl_size);

    return 0;

dma_alloc_attrs_failed:
is_device_dma_capable_failed:
    return -1;
}

static size_t
get_channel_count_v0_0(struct enyx_hfp_tx_dev * dev)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    return impl->channel_count;
}

static size_t
get_mtu_v0_0(struct enyx_hfp_tx_dev * dev, uint16_t channel_id)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    struct enyx_hfp_tx_channel * channel = &impl->channels[channel_id];
    /* Due to a hardware bug, the whole message size must be
     * less or equal to the reported MTU. Hence return the hardware MTU
     * minus the maximu metadata size (i.e. enyx_hfp header + padding) */
    return channel->mtu - sizeof(union enyx_hfp_tx_msg_hdr) - impl->coherency_line_size;
}

static size_t
get_mtu_v1_2(struct enyx_hfp_tx_dev * dev, uint16_t channel_id)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    struct enyx_hfp_tx_channel * channel = &impl->channels[channel_id];
    return channel->mtu;
}

static size_t
get_ctrl_size_v0_0(struct enyx_hfp_tx_dev * dev)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    return impl->ctrl_size;
}

static size_t
get_buffer_size_v0_0(struct enyx_hfp_tx_dev * dev, uint16_t channel_id)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    struct enyx_hfp_tx_channel * channel = &impl->channels[channel_id];
    return channel->buffer_size;
}

static int
send_v0_0(struct enyx_hfp_tx_dev * dev,
                uint16_t channel_id,
                const void * data_ptr,
                uint32_t data_size)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    struct enyx_hfp_tx_channel * c = &impl->channels[channel_id];
    const uint64_t * src, * src_end;
    uint64_t * dst, * dst_end;
    uint16_t msg_size;
    union enyx_hfp_tx_msg_hdr h = {
        .b = {0},
    };

    msg_size = ALIGN(sizeof(h) + data_size, impl->coherency_line_size);

    if (data_size > impl->methods.get_mtu(dev, channel_id))
        return -EINVAL;

    /* Wait for room on card buffer but do not consume the entire buffer
     * in order to prevent rd and wr pointers to be equal when the
     * buffer is full */
    if (c->buffer_free_size <= msg_size) {
        const int32_t rd_to_wr = c->buffer_offset - *c->buffer_rd_ptr;
        if (rd_to_wr < 0)
            c->buffer_free_size = - rd_to_wr;
        else
            c->buffer_free_size = c->buffer_size - rd_to_wr;

        if (c->buffer_free_size <= msg_size)
            return -EAGAIN;
    }

    dst = (void *)&c->buffer_start[c->buffer_offset];
    dst_end = (void *)&c->buffer_start[c->buffer_size];

    /* Send header */
    h.s.data_size = data_size;

    writeq(h.b[0], dst++);
    writeq(h.b[1], dst++);
    writeq(h.b[2], dst++);
    writeq(h.b[3], dst++);

    src = data_ptr;
    /* We may have a buffer overrun read here */
    src_end = src + ALIGN(data_size, sizeof(*src)) / 8;

    /* Send data */
    while (src != src_end) {
        writeq(*src++, dst++);
        if (dst == dst_end)
            dst = (void *)&c->buffer_start[0];
    }

    /* Send padding */
    while ((uintptr_t)dst & (impl->coherency_line_size - 1))
        /* dst buffer rollover check is not required as we won't
           write more than what's required to be aligned
           and the end of the buffer is aligned... */
        writeq(0, dst++);

    c->buffer_offset = (c->buffer_offset + msg_size) % c->buffer_size;
    c->buffer_free_size -= msg_size;

    return 0;
}

static size_t
get_usage_v0_1(struct enyx_hfp_tx_dev * dev,
               uint16_t channel_id,
               char * buffer, size_t buffer_size)
{
    struct enyx_hfp_tx_impl * impl = dev_to_enyx_hfp_tx_impl(dev);
    size_t count = 0;

    if (down_interruptible(&impl->parent->lock))
        return 0ULL;

    enyx_hfp_write8(impl->parent, HFP_TX_INFO_DEVICE_ID, channel_id);

    count = enyx_hfp_read_str_unlocked(impl->parent,
                                  HFP_TX_INFO_USAGE_STR,
                                  HFP_TX_INFO_USAGE_STR_INDEX,
                                  HFP_TX_INFO_PAGE_READY,
                                  buffer, buffer_size);

    up(&impl->parent->lock);
    return count;
}

static int
reset(struct enyx_hfp_tx_impl * impl)
{
    enyx_hfp_write8(impl->parent, HFP_TX_RESET, 1);

    if (enyx_hfp_wait_until_val_is(impl->parent, HFP_TX_RESET, 0) < 0) {
        dev_err(&impl->parent->device, "Failed to reset device\n");
        goto reset_cmd_failed;
    }

    return 0;

reset_cmd_failed:
    return -1;
}

static void
bind_methods(struct enyx_hfp_tx_impl * impl)
{
    impl->methods.enable_channel = enable_channel_v0_0;
    impl->methods.enable_all_channels = enable_all_channels_v0_0;
    impl->methods.disable_channel = disable_channel_v0_0;
    impl->methods.disable_all_channels = disable_all_channels_v0_0;
    impl->methods.mmap_ctrl = mmap_ctrl_v0_0;
    impl->methods.mmap_buffer = mmap_buffer_v0_0;
    impl->methods.get_channel_count = get_channel_count_v0_0;
    impl->methods.get_mtu = get_mtu_v0_0;
    impl->methods.get_ctrl_size = get_ctrl_size_v0_0;
    impl->methods.get_buffer_size = get_buffer_size_v0_0;
    impl->methods.send = send_v0_0;

    if (enyx_hfp_get_device_version(impl->parent) >= HFP_COMPUTE_VERSION(1, 1))
        impl->methods.get_usage = get_usage_v0_1;

    if (enyx_hfp_get_device_version(impl->parent) >= HFP_COMPUTE_VERSION(1, 2))
        impl->methods.get_mtu = get_mtu_v1_2;
}

static void init_channels_v0_0(struct enyx_hfp_tx_impl * impl) {
    uint16_t i;
    for (i = 0; i < impl->channel_count; i++) {
        struct enyx_hfp_tx_channel * channel = &impl->channels[i];
        channel->mtu = impl->global_mtu;
        channel->buffer_size = impl->buffer_increment;
    }

}

static void init_channels_v2_0(struct enyx_hfp_tx_impl * impl)
{
    uint16_t i;
    uint32_t shift;
    for (i = 0; i < impl->channel_count; i++) {
        struct enyx_hfp_tx_channel * channel = &impl->channels[i];
        channel->mtu = get_info_16_v0_2(impl, i, HFP_TX_DEVICE_MTU);
        shift = get_info_16_v0_2(impl, i, HFP_TX_DEVICE_BUFFER_SIZE);
        channel->buffer_size = channel->buffer_free_size = 1ULL << shift;
    }
}

struct enyx_hfp_tx_dev *
enyx_hfp_tx_dev_create(struct enyx_hfp_device * parent,
                  size_t coherency_line_size)
{
    uint16_t channel_count;
    struct enyx_hfp_tx_impl * impl;

    if (! get_device(&parent->device))
        goto get_device_failed;

    if (parent->size < 14 * 4) {
        dev_err(&parent->device,
                "Reported device memory range smaller than expected\n");
        goto size_check_failed;
    }

    if (parent->id.version_major > 2) {
        dev_err(&parent->device,
                "Reported device major version %hhu is incompatible\n",
                parent->id.version_major);
        goto major_check_failed;
    }

    channel_count = enyx_hfp_read8(parent, HFP_TX_DEVICE_COUNT);

    impl = kzalloc(sizeof(*impl) + sizeof(*impl->channels) * channel_count, GFP_KERNEL);
    if (! impl) {
        dev_err(&parent->device, "Can't allocate tx\n");
        goto kzalloc_failed;
    }

    impl->parent = parent;
    impl->channel_count = channel_count;
    impl->global_mtu = enyx_hfp_read16(parent, HFP_TX_MTU);

    impl->buffer_phys_addr = impl->parent->bus->phys_addr
            + enyx_hfp_read32(parent, HFP_TX_BUFFER_BASE_OFFSET_DW) * 4;
    impl->buffer_increment = enyx_hfp_read32(parent, HFP_TX_BUFFER_INCREMENT);

    dev_dbg(&parent->device,
            "Buffer(s) located at addr 0x%llx every %zu bytes\n",
            impl->buffer_phys_addr, impl->buffer_increment);

    /* Initialize buffer size and mtu */
    init_channels_v0_0(impl);
    if (enyx_hfp_get_device_version(impl->parent) >= HFP_COMPUTE_VERSION(2, 0))
        init_channels_v2_0(impl);

    impl->buffer_size = impl->channel_count * impl->buffer_increment;

    dev_dbg(&parent->device, "Found %u channels\n", impl->channel_count);

    if (reset(impl))
        goto reset_failed;

    if (setup_cache_line(impl, coherency_line_size) < 0)
        goto setup_cache_line_failed;

    impl->buffer_io_addr = ioremap_wc(impl->buffer_phys_addr,
                                      impl->buffer_size);
    if (! impl->buffer_io_addr) {
        dev_err(&parent->device, "Can't map buffer\n");
        goto ioremap_wc_failed;
    }

    /* Setup contigous DMA ctrls ASAP to ensure
     * memory fragmentation won't be an issue while
     * device will be open later */
    if (alloc_dma_ctrl(impl) < 0)
        goto alloc_dma_ctrl_failed;

    bind_methods(impl);

    return &impl->methods;

alloc_dma_ctrl_failed:
setup_cache_line_failed:
    iounmap(impl->buffer_io_addr);
ioremap_wc_failed:
reset_failed:
    kfree(impl);
kzalloc_failed:
major_check_failed:
size_check_failed:
    put_device(&parent->device);
get_device_failed:
    return NULL;
}
EXPORT_SYMBOL(enyx_hfp_tx_dev_create);


