#pragma once
#include <enyx/hw/a2c_stream.hpp>
#include <enyx/hw/c2a_stream.hpp>
#include <enyx/cores/data_stream/hw_source.hpp>
#include <enyx/cores/data_stream/hw_sink.hpp>
#include <enyx/cores/tcp/tcp.hpp>
#include <enyx/cores/tcp/context.hpp>
#include <enyx/cores/tcp/data_stream.hpp>
#include <enyx/fix/session.hpp>
#include <enyx/fix/message.hpp>
#include <enyx/fix/tcp.hpp>
#include <enyx/fix/protocol.hpp>

namespace enyx::fix {

class Acceptor;

namespace EventType {

/**
 * @ref FixSourceHeader message type
 */
enum EventType {
    /// The FIX engine sent a message to a peer
    TxMsgEvent = 0,
    /// The FIX engine received a message from a peer
    RxMsgEvent,
    /**
     * The FIX engine forwards a valid @ref Message to the @ref
     * Acceptor.
     */
    RxValidMsg,
    /**
     * The FIX engine detected an error and forwards the corresponding @p
     * Message to the @ref Acceptor in order to perform additional checks and
     * respond appropriately.
     */
    RxErrMsg,
    /**
     * The FIX engine detected an error on a message received from a peer.
     */
    RxError,
    /**
     * The FIX engine detected an error on a message that would have been sent
     * to a peer.
     */
    TxError,
};

};

enum class ErrorCode : std::uint8_t {
    SUCCESS = 0,
    TCP_SESSION_STALLED,
    DUPLICATED_TAG,
    VALUE_VALIDATION,
    VALUE_CONVERSION,
    UNKNOWN_FIX_TAG,
    MISSING_TAGVALUE_IN_MSG,
    UNKNOWN_MSGTYPE,
    INVALID_FIX_TAG,
    MSGSEQNUM_DUP,
    MSGSEQNUM_GAP,
    INVALID_SENDERCOMPID,
    INVALID_TARGETCOMPID,
    MSGSEQNUM_MISSING,
    UNKNOWN_BEGIN_STRING,
    INVALID_FIX_CHECKSUM,
    NO_FIX_CHECKSUM,
    MSG_NOT_SIZE_OF_BODYLENGTH,
    INVALID_FIX_HDR,
    INVALID_FCS,
};

namespace {

struct FixCategory : std::error_category
{
    const char *
    name() const noexcept override;

    std::string
    message(int ev) const override;
};

}

const std::error_category &
fix_category() noexcept;

struct FixSource {
    FixSource(Acceptor & acceptor);

    struct [[gnu::packed]] metadata {
        data_stream::metadata metadata;
        FixSourceHeader header;
    };

    static std::tuple<Message, std::size_t>
    BinToMessage(std::uint8_t const * data,
                 std::size_t size);

    void
    operator()(std::uint8_t const * data,
               std::size_t size,
               void const * metadata);

    Acceptor & acceptor_;
};

struct FixSink {
    FixSink(data_stream::sink & sink);

    struct [[gnu::packed]] metadata {
        data_stream::metadata metadata;
        SinkHeader header;
    };

    static std::vector<send_buffer>
    MessageToVectors(Message const & message);

    int
    send(data_stream::sink::buffer const * buffer, void const * metadata);

    static int
    send_io_vec(::enyx_data_stream_sink const* next_sink,
                void * impl,
                ::enyx_data_stream_sink_buffer const * buffer,
                void const * metadata);

    static ssize_t
    get_mtu(struct ::enyx_data_stream_sink const * sink);

    static ::enyx_data_stream_sink
    _make_sink(::enyx_data_stream_sink * next_sink);

    data_stream::sink & next_sink_;
    data_stream::sink sink_;
};

struct AnyEventSource {
    AnyEventSource(FixSource & fix_source,
                   tcp::context & context);

    struct metadata {
        enum class EventType {
            TCP = 0,
            FIX = 1,
        };
        data_stream::metadata metadata;
        EventType type;
    };

    void
    operator()(std::uint8_t const * data,
               std::size_t size,
               void const * next_metadata);

    FixSource & fix_source_;
    tcp::context & context_;
};

/**
 * Stream wrapper for fix
 *
 * This is currently designed to work over a FIX stack
 *
 * Part of the implementation
 */
class LowLevelTransport {
public:
    LowLevelTransport(TcpSessionManager & tcp_session_manager,
                      Acceptor & acceptor,
                      enyx::hw::a2c_stream && data_a2c,
                      enyx::hw::c2a_stream && data_c2a);

    void
    poll();

    ssize_t
    send(data_stream::sink::buffer const * buffer, void const * metadata);

private:
    TcpSessionManager & tcp_session_manager_;
    Acceptor & acceptor_;
    enyx::hw::a2c_stream a2c_;
    enyx::hw::c2a_stream c2a_;
    FixSource fix_source_;
    enyx::data_stream::sink raw_sink_;
    FixSink fix_sink_;
    enyx::tcp::context context_;
    AnyEventSource event_source_;
    enyx::data_stream::hw_source raw_source_;
};
}
