#pragma once
#include <limits>
#include <string>

namespace enyx {
namespace fix {

/// FIX session
struct Session {
    using PeerCompID = std::string;

    Session()
        : sent_logon{false}
        , received_logon{false}
        , sent_logout{false}
        , received_logout{false}
        , sent_reset{false}
        , received_reset{false}
        , current_test_request{0}
        , heart_bt_int(std::numeric_limits<std::uint32_t>::max())
        , hb_in_progress{false}
        , last_sent_time{}
        , last_received_time{}
        , resend_range{}
    {}

    void reset()
    {
        sent_logon = false;
        received_logon = false;
        sent_logout = false;
        received_logout = false;
        sent_reset = false;
        current_test_request = false;
        heart_bt_int = std::numeric_limits<std::uint32_t>::max();
        last_sent_time = FieldTypes::UTCTimestamp{};
        last_received_time = FieldTypes::UTCTimestamp{};
        resend_range = {};
    }

    bool need_heartbeat()
    {
        if (hb_in_progress)
            return true;
        if (current_test_request != 0)
            return false;
        auto now = FieldTypes::UTCTimestamp{}.value().tv_sec;
        if (current_test_request != 0)
            return false;;
        return (now - last_sent_time.value().tv_sec) >= heart_bt_int;
    }

    bool need_test_request()
    {
        auto now = FieldTypes::UTCTimestamp{}.value().tv_sec;
        auto const max_delay = 1.2 * (current_test_request + 1) * heart_bt_int;
        return (now - last_received_time.value().tv_sec) >= max_delay;
    }

    bool connection_timed_out()
    {
        auto now = FieldTypes::UTCTimestamp{}.value().tv_sec;
        auto const max_delay = 2.4 * heart_bt_int;
        return (now - last_received_time.value().tv_sec) >= max_delay;
    }

    bool resend_requested()
    {
        auto & [begin, end] = resend_range;
        return begin.value() != 0;
    }

    void
    disarm_resend()
    {
        resend_range = {};
    }

    void
    trigger_resend(MsgSeqNum begin, MsgSeqNum end = MsgSeqNum{0})
    {
        resend_range = {begin, end};
    }

    bool sent_logon;
    bool received_logon;
    bool sent_logout;
    bool received_logout;
    bool sent_reset;
    bool received_reset;
    std::uint32_t current_test_request;
    std::uint32_t heart_bt_int;
    bool hb_in_progress;
    FieldTypes::UTCTimestamp last_sent_time;
    FieldTypes::UTCTimestamp last_received_time;
    std::tuple<MsgSeqNum, MsgSeqNum> resend_range;
};

/// TCP/FIX connection
struct Connection {
    std::uint32_t tcp_session_id{0};
    std::uint32_t fix_session_id{0};
    std::string target_comp_id{""};
    Session state;
};

/// Internal structure for asynchronous closing
struct TcpSessionToClose {
    std::uint32_t tcp_session_id;
    FieldTypes::UTCTimestamp closing_due_date;
};


}
}
