#pragma once

#include <cstdint>

#include <array>
#include <optional>
#include <string>
#include <unordered_map>
#include <variant>
#include <vector>

namespace enyx::fix::model {

/**
 * This module defines fix model structures and a function to load it from a
 * YAML file.
 */

// Internal field types
enum class field_type {
    AMT,
    BOOLEAN,
    CHAR,
    CURRENCY,
    FLOAT,
    INT,
    LENGTH,
    LOCALMKTDATE,
    MULTIPLEVALUESTRING,
    NUMINGROUP,
    PERCENTAGE,
    PRICE,
    PRICEOFFSET,
    QTY,
    SEQNUM,
    STRING,
    TAGNUM,
    UTCDATEONLY,
    UTCTIMESTAMP,

    // Not supported: YAML string types unmapped
    /* COUNTRY, */
    /* DATA, */
    /* DAYOFMONTH, */
    /* EXCHANGE, */
    /* MONTHYEAR, */
    /* UTCTIMEONLY, */
};

// Map YAML type strings to internal field types
static const std::unordered_map<std::string, field_type> field_name_to_type = {
    {"Amt",                 field_type::AMT},
    {"Boolean",             field_type::BOOLEAN},
    {"Currency",            field_type::CURRENCY},
    {"Float",               field_type::FLOAT},
    {"Length",              field_type::LENGTH},
    {"LocalMktDate",        field_type::LOCALMKTDATE},
    {"MultipleValueString", field_type::MULTIPLEVALUESTRING},
    {"NumInGroup",          field_type::NUMINGROUP},
    {"Percentage",          field_type::PERCENTAGE},
    {"Price",               field_type::PRICE},
    {"PriceOffset",         field_type::PRICEOFFSET},
    {"Qty",                 field_type::QTY},
    {"SeqNum",              field_type::SEQNUM},
    {"String",              field_type::STRING},
    {"TagNum",              field_type::TAGNUM},
    {"UTCDateOnly",         field_type::UTCDATEONLY},
    {"UTCTimestamp",        field_type::UTCTIMESTAMP},
    {"char",                field_type::CHAR},
    {"int",                 field_type::INT},
};

namespace validator {

struct accepted_values {
    std::vector<char> values;
};

struct accepted_ranges {
    std::array<char, 2> range0;
    std::array<char, 2> range1;
};

struct max_length {
    std::uint32_t max_length;
};

struct zero {
};

}
using validator_t = std::variant<
        validator::accepted_values,
        validator::accepted_ranges,
        validator::max_length,
        validator::zero>;

struct field;
struct field_def;

using field_list = std::vector<field>;
using field_defs = std::unordered_map<std::string, field_def>;

struct repeatable_group {
    field_def const & separator;
    field_list fields;
};

struct field_def {
    std::string name;
    std::uint32_t tag;
    field_type type;

    std::optional<validator_t> validator;

    std::optional<repeatable_group> repeated;
};

struct field {
    field_def const & def;
    bool mandatory;
    bool forward;
};

struct message {
    std::string name;
    std::string msgtype;
    field_list fields;
};

using messages = std::vector<message>;

struct currency {
    char code[3];
    std::uint16_t value;
};

using currencies = std::vector<currency>;

struct fix_model {
    field_defs const fields;
    field_list common_fields;
    std::vector<message> messages;
    std::vector<currency> currencies;
    std::uint8_t previous_months;
    field_list const & std_fields;
    std::vector<message> const & admin_messages;
};

fix_model
load_yaml_model(std::string const & file);

}
