#include <boost/program_options.hpp>

#include <fstream>
#include <iostream>
#include <iterator>

#include <stdexcept>
#include <stdio.h>

#include <enyx/cores/transceiver/transceiver.hpp>

#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/core_tree.hpp>
#include <enyx/hw/core.hpp>

#include "../lib/DeviceDiscovery.hpp"

namespace po = boost::program_options;

struct run_params {
    std::string accelerator = "0";
    bool show = false;
    ssize_t sfp = -1;
    ssize_t qsfp = -1;
    bool reset = false;
    int i2c = -1;
    int lp = -1;
    int rate = -1;
    int tx = -1;
};

void
dump_sfp_info(enyx::transceiver::sfp const & sfp)
{
    std::cout << "SFP  " << (unsigned) sfp.get_id().v() << " status" << std::endl;

    if (sfp.is_transmitter_enabled().v())
        std::cout << "\t transmitter is enabled" << std::endl;
    else
        std::cout << "\t transmitter is disabled" << std::endl;

    if (sfp.get_tx_fault().v())
        std::cout << "\t TX fault detected" << std::endl;
    else
        std::cout << "\t no TX fault" << std::endl;

    if (sfp.has_signal().v())
        std::cout << "\t signal detected" << std::endl;
    else
        std::cout << "\t no signal detected" << std::endl;

    if (sfp.get_rate().v() == ENYX_SFP_FULL_BANDWIDTH)
        std::cout << "\t full bandwidth mode" << std::endl;
    else
        std::cout << "\t reduced bandwidth mode" << std::endl;
}

void
dump_qsfp_info(enyx::transceiver::qsfp & qsfp)
{
    std::cout << "QSFP " << (unsigned) qsfp.get_id().v() << " status" << std::endl;

    if (qsfp.is_low_power_mode_enabled().v())
        std::cout << "\t low power mode enabled" << std::endl;
    else
        std::cout << "\t low power mode disabled" << std::endl;

    if (qsfp.is_i2c_communication_enabled().v())
        std::cout << "\t I2C communication enabled" << std::endl;
    else
        std::cout << "\t I2C communication disabled" << std::endl;

    if (qsfp.possible_fault_detected().v())
        std::cout << "\t possible fault detected" << std::endl;
    else
        std::cout << "\t no fault detected" << std::endl;
}

void
run_transceiver_command(run_params const & params)
{
    auto accelerator = enyx::create_accelerator(params.accelerator);
    auto mmio = enyx::get_first_mmio(accelerator);
    auto core_tree = enyx::hw::enumerate_cores(mmio);
    auto root = core_tree.get_root();
    auto transceiver = enyx::transceiver::transceiver(root);

    if (params.show) {
        auto sfp_count = transceiver.count_sfp().v();
        std::cout << "Found " << sfp_count << " SFP ports" << std::endl;
        for (size_t i = 0; i < sfp_count; i++)
            dump_sfp_info(transceiver.get_sfp(i).v());

        auto qsfp_count = transceiver.count_qsfp().v();
        std::cout << "Found " << qsfp_count << " QSFP ports" << std::endl;
        for (size_t i = 0; i < qsfp_count; i++) {
            auto qsfp = transceiver.get_qsfp(i);
            if (qsfp)
                dump_qsfp_info(qsfp.v());
            else
                std::cout << "No QSFP plugged on port " << i << std::endl;
        }

        return;
    }

    if (params.qsfp >= 0) {
        auto qsfp = transceiver.get_qsfp(params.qsfp).v();
        if (params.reset) {
            std::cout << "Resetting QSFP " << params.qsfp << std::endl;
            qsfp.reset_module();

            return;
        }

        if (params.i2c == 0) {
            std::cout << "Disabling I2C communication on QSFP " << params.qsfp
                      << std::endl;
            qsfp.disable_i2c_communication();
            return;
        }
        if (params.i2c == 1) {
            std::cout << "Enabling I2C communication on QSFP " << params.qsfp
                      << std::endl;
            qsfp.enable_i2c_communication();
            return;
        }

        if (params.lp == 0) {
            std::cout << "Disabling low power mode on QSFP " << params.qsfp
                      << std::endl;
            qsfp.disable_low_power_mode();
            return;
        }
        if (params.lp == 1) {
            std::cout << "Enabling low power mode on QSFP " << params.qsfp
                      << std::endl;
            qsfp.enable_low_power_mode();
            return;
        }
    }

    if (params.sfp >= 0) {
        auto sfp = transceiver.get_sfp(params.sfp).v();

        if (params.rate == 0) {
            std::cout << "Setting reduced bandwidth mode on SFP " << params.sfp
                      << std::endl;
            sfp.set_rate(ENYX_SFP_REDUCED_BANDWIDTH);
            return;
        }
        if (params.rate == 1) {
            std::cout << "Setting full bandwidth mode on SFP " << params.sfp
                      << std::endl;
            sfp.set_rate(ENYX_SFP_FULL_BANDWIDTH);
            return;
        }

        if (params.tx == 0) {
            std::cout << "Disabling TX on SFP " << params.sfp
                      << std::endl;
            sfp.disable_transmitter();
            return;
        }
        if (params.rate == 1) {
            std::cout << "Enabling TX on SFP " << params.sfp
                      << std::endl;
            sfp.enable_transmitter();
            return;
        }
    }
}

run_params
handle_options(po::variables_map args)
{
    run_params params;

    if (! args.count("accelerator")) {
        throw std::runtime_error("--accelerator option is mandatory");
    }
    params.accelerator = args["accelerator"].as<std::string>();

    if (args.count("show")) {
        params.show = true;
    }

    if (! params.show && (args.count("sfp") == args.count("qsfp"))) {
        throw std::runtime_error("Either --sfp or --qsfp must be specified");
    }

    if (args.count("qsfp")) {
        params.qsfp = args["qsfp"].as<size_t>();

        if (args.count("reset")) {
            params.reset = true;
        }

        if (args.count("i2c")) {
            if (args["i2c"].as<std::string>() == "enable")
                params.i2c = 1;
            else if (args["i2c"].as<std::string>() == "disable")
                params.i2c = 0;
            else
                throw std::runtime_error("--i2c must be either enable or disable");
        }

        if (args.count("low-power")) {
            if (args["low-power"].as<std::string>() == "enable")
                params.lp = 1;
            else if (args["low-power"].as<std::string>() == "disable")
                params.lp = 0;
            else
                throw std::runtime_error("--low-power must be either enable or disable");
        }
    }

    if (args.count("sfp")) {
        params.qsfp = args["sfp"].as<size_t>();

        if (args.count("tx")) {
            if (args["tx"].as<std::string>() == "enable")
                params.tx = 1;
            else if (args["tx"].as<std::string>() == "disable")
                params.tx = 0;
            else
                throw std::runtime_error("--tx must be either enable or disable");
        }

        if (args.count("set-rate")) {
            if (args["set-rate"].as<std::string>() == "full")
                params.rate = 1;
            else if (args["set-rate"].as<std::string>() == "reduced")
                params.rate = 0;
            else
                throw std::runtime_error("--rate must be either full or reduced");
        }
    }

    return params;
}

po::variables_map
parse(int ac, char* av[])
{
    po::options_description desc("Allowed options");
    desc.add_options()
        ("help,h", "produce help message")
        ("accelerator,a",
         po::value<std::string>(), "accelerator index or name")
        ("show,s", "show all SFP/QSFP info")
        ("sfp", po::value<uint32_t>(), "SFP index")
        ("qsfp", po::value<size_t>(), "SFP index")
        ("reset", "Reset QSFP")
        ("i2c", po::value<std::string>(), "enable/disable I2C on QSFP")
        ("low-power", po::value<std::string>(), "enable/disable low power mode on QSFP")
        ("tx", po::value<std::string>(), "enable/disable TX on SFP")
        ("set-rate", po::value<std::string>(), "set SFP bandwidth rate")
        ;

    po::variables_map args;

    po::store(po::parse_command_line(ac, av, desc), args);
    po::notify(args);

    if (args.count("help")) {
        std::cout << desc << std::endl;
        exit(0);
    }

    return args;
}

int main(int ac, char* av[])
{
    auto args = parse(ac, av);

    auto params = handle_options(args);
    run_transceiver_command(params);

    return 0;
}
