#include "Size.hpp"

#include <stdexcept>
#include <iostream>
#include <sstream>
#include <iomanip>

#include <boost/io/ios_state.hpp>
#include <boost/regex.hpp>

#include <enyx/cores/namespace.hpp>

ENYX_CORES_NAMESPACE_BEGIN

namespace {

Size
parse_size(const std::string & s)
{
    boost::regex regex_iec(R"((\d+)\s*([KMGTPE]i)?(B|bit))");
    boost::regex regex_si(R"((\d+)\s*([kKMGTPE])?(B|b|bit))");
    boost::smatch m;

    Size::UnitSystem unit_system;
    if (boost::regex_match(s, m, regex_iec))
        unit_system = Size::IEC;
    else if (boost::regex_match(s, m, regex_si))
        unit_system = Size::SI;
    else
    {
        std::ostringstream error;
        error << "size '" << s
              << "' doesn't match either \\d+\\s*([KMGTPE]i)?(B|bit)"
                 " nor \\d+\\s*([kKMGTPE])?(B|b|bit)";
        throw std::runtime_error(error.str());
    }

    const uint64_t factor = unit_system == Size::SI ? 1000 : 1024;

    uint64_t size = std::atoll(m.str(1).c_str());

    if (m[2].matched)
        switch (m.str(2)[0])
        {
            case 'E':
                size *= factor;
                // Fall through
            case 'P':
                size *= factor;
                // Fall through
            case 'T':
                size *= factor;
                // Fall through
            case 'G':
                size *= factor;
                // Fall through
            case 'M':
                size *= factor;
                // Fall through
            case 'K':
            case 'k':
                size *= factor;
        }

    if (m.str(3) == "bit" || m.str(3) == "b")
        size /= 8;

    return Size{size, unit_system};
}

const char * units_iec[] = { "bit", "Kibit", "Mibit",
                             "Gibit", "Tibit", "Pibit",
                             "Eibit" };

const char * units_si[] = { "bit", "kbit", "Mbit",
                            "Gbit", "Tbit", "Pbit",
                            "Ebit" };

#define UNITS_COUNT ((sizeof(units_iec)) / sizeof(units_iec[0]))


} // anonymous namespace

std::istream &
operator>>(std::istream & in, Size & size)
{
    std::istream::sentry sentry(in);

    if (sentry)
    {
        std::string s;
        in >> s;

        size = parse_size(s);
    }

    return in;
}


std::ostream &
operator<<(std::ostream & out, const Size & size)
{
    std::ostream::sentry sentry(out);
    if (sentry)
    {
        const Size::UnitSystem unit_system = size.get_unit_system();
        const uint64_t factor = unit_system == Size::SI ? 1000 : 1024;
        const char ** units = unit_system == Size::SI ? units_si : units_iec;

        long double value = size * 8;
        uint64_t i;
        for (i = 0; i != UNITS_COUNT - 1 && value / factor >= 1.; ++i)
            value /= factor;

        boost::io::ios_flags_saver s(out);
        out << std::fixed << std::setprecision(1) << value << units[i];
    }

    return out;
}

ENYX_CORES_NAMESPACE_END
