#include <iostream>
#include <iomanip>
#include <sstream>
#include <cstdint>
#include <vector>
#include <algorithm>
#include <enyx/cores/namespace.hpp>

ENYX_CORES_NAMESPACE_BEGIN

struct Percentile final
{
    double value;
};

std::ostream &operator<<(std::ostream & os, Percentile percentile)
{
    std::ios_base::fmtflags flags{os.flags()};
    os << std::fixed << std::setprecision(1)
       << std::setw(12) << percentile.value;
    os.flags(flags);
    return os;
}

class Statistics final
{
public:
    Statistics(std::uint64_t packet_count, double cpu_freq)
        : sorted_{false}
        , packet_count_(packet_count)
        , latencies_(packet_count_)
        , cpu_freq_(cpu_freq)
    {}

    void
    add_point(std::uint64_t packet_id, std::uint64_t value)
    {
        if (sorted_)
            throw std::runtime_error{"Cannot insert point in sorted stats"};
        latencies_.at(packet_id) = value;
    }

    void
    sort()
    {
        std::sort(latencies_.begin(), latencies_.end());
        sorted_ = true;
    }

    double
    get_percentile(double percentile)
    {
        if (! sorted_)
            throw std::runtime_error{"Cannot get point in raw stats"};
        double packet_index = packet_count_ * percentile / 100.0;
        return latencies_[static_cast<std::uint64_t>(packet_index)] / cpu_freq_;
    }

    double
    get_max()
    {
        if (! sorted_)
            throw std::runtime_error{"Cannot get point in raw stats"};
        return latencies_[packet_count_ - 1] / cpu_freq_;
    }

private:
    bool sorted_;
    std::uint64_t packet_count_;
    std::vector<std::uint64_t> latencies_;
    double cpu_freq_;
};

ENYX_CORES_NAMESPACE_END
