#include "ArpScope.hpp"

#include <sstream>
#include <stdexcept>
#include <iomanip>

#include <enyx/cores/hardware_ids.hpp>

#include "Color.hpp"

ENYX_CORES_NAMESPACE_BEGIN

namespace ArpScope {

namespace {

namespace en = enyx;
namespace na = en::arp;

na::arp
create_arp(hw::core & root,
           Application::StackType stack_type,
           std::uint32_t stack_index)
{
    std::vector<na::arp> arp;

    for (auto & core : root)
    {
        auto const hardware_id = core.get_descriptor().hardware_id;
        if (hardware_id == enyx::hardware_ids::TCP_STD)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::TCP ||
                stack_type == Application::StackType::TCP_STD)
                arp.emplace_back(core);
        }

        else if (hardware_id == enyx::hardware_ids::UDP_STD)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::UDP ||
                stack_type == Application::StackType::UDP_STD)
                arp.emplace_back(core);
        }

        else if (hardware_id == enyx::hardware_ids::TCP_ULL)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::TCP ||
                stack_type == Application::StackType::TCP_ULL)
                arp.emplace_back(core);
        }

        else if (hardware_id == enyx::hardware_ids::UDP_ULL)
        {
            if (stack_type == Application::StackType::ALL ||
                stack_type == Application::StackType::UDP ||
                stack_type == Application::StackType::UDP_ULL)
                arp.emplace_back(core);
        }
    }

    if (arp.size() <= stack_index)
    {
        std::ostringstream error;
        error << "Expected at least " << (stack_index + 1)
              << " stack(s)";
        throw std::runtime_error{error.str()};
    }

    return std::move(arp[stack_index]);
}

char const USAGE[] =
"Usage: " BIN_NAME " arp show\n"
"       " BIN_NAME " arp set server ( enable | disable )\n"
"       " BIN_NAME " arp set reply_timeout TIMEOUT_MS\n"
"       " BIN_NAME " arp table show\n"
"       " BIN_NAME " arp table add IP_ADDR MAC_ADDR\n"
"       " BIN_NAME " arp table delete IP_ADDR\n"
"       " BIN_NAME " arp table find IP_ADDR\n"
"       " BIN_NAME " arp table set lifetime LIFETIME_S\n"
"       " BIN_NAME " arp table flush [ dynamic ]\n";

void
set_server_enable(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    ensure_end(i, e);

    arp.enable_server().value();
}

void
set_server_disable(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    ensure_end(i, e);

    arp.disable_server().value();
}

void
set_server(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    if (i == e)
        throw std::runtime_error{"Expected { enable | disable }"};

    if (is_prefix(*i, "enable"))
        set_server_enable(++i, e, arp);
    else if (is_prefix(*i, "disable"))
        set_server_disable(++i, e, arp);
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
            << "', try '" BIN_NAME " arp help'";
        throw std::runtime_error{error.str()};
    }
}

void
set_reply_timeout(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    if (i == e)
        throw std::runtime_error{"Expected TIMEOUT_MS"};

    std::uint16_t const timeout = std::atoi(*i);

    ensure_end(++i, e);

    arp.set_client_timeout(timeout).value();
}

void
set(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    if (i == e)
        throw std::runtime_error{"Expected { server | reply_timeout }"};

    if (is_prefix(*i, "server"))
    {
        set_server(++i, e, arp);
    }
    else if (is_prefix(*i, "reply_timeout"))
    {
        set_reply_timeout(++i, e, arp);
    }
    else
    {
        std::ostringstream error;
        error <<  "Unknown attribute '" << *i
              << "', try '" BIN_NAME " arp help'";
        throw std::runtime_error{error.str()};
    }
}

void
table_add(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    if (i == e)
        throw std::runtime_error{"Expected IP_ADDR"};

    en::ipv4_address ipv4{*i++};

    if (i == e)
        throw std::runtime_error{"Expected MAC_ADDR"};

    en::mac_address const mac{*i};

    ensure_end(++i, e);

    arp.table_add_entry({ipv4, mac}).value();
}

void
table_delete(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    if (i == e)
        throw std::runtime_error{"Expected IP_ADDR"};

    en::ipv4_address const ipv4{*i};

    ensure_end(++i, e);

    arp.table_delete_entry(ipv4).value();
}

void
table_find(CmdLineIterator i, CmdLineIterator e,  na::arp & arp)
{
    if (i == e)
        throw std::runtime_error{"Expected IP_ADDR"};

    en::ipv4_address const ipv4{*i};

    ensure_end(++i, e);

    auto mac = arp.table_find_mac(ipv4).value();

    std::cout << mac << std::endl;
}

void
table_set_lifetime(CmdLineIterator i, CmdLineIterator e,  na::arp & arp)
{
    if (i == e)
        throw std::runtime_error{"Expected LIFETIME_S"};

    std::uint16_t const lifetime = std::atoi(*i);

    ensure_end(++i, e);

    arp.set_dynamic_lifetime(lifetime).value();
}

void
table_flush(CmdLineIterator i, CmdLineIterator e,  na::arp & arp)
{
    if (i == e)
    {
        auto count = arp.table_get_max_entry_count();
        for (std::uint16_t it = 0; it < count; ++it)
        {
            auto result = arp.table_get_entry(it);
            if (! result)
                continue;

            arp.table_delete_entry(result.value().ipv4).value();
        }
    }
    else if (is_prefix(*i, "dynamic"))
    {
        ensure_end(++i, e);

        arp.table_flush_dynamic_entries().value();
    }
    else
    {
        std::ostringstream error;
        error <<  "Unknown ACTION '" << *i << "', try '" BIN_NAME " arp help'";
        throw std::runtime_error{error.str()};
    }
}

void
table_show(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    ensure_end(i, e);

    std::cout << "dynamic entry lifetime: "
              << arp.get_dynamic_lifetime().value() << '\n';
    auto count = arp.table_get_max_entry_count();
    std::cout << "arp table:" << '\n';
    for (std::uint16_t it = 0; it != count; ++it)
    {
        auto result = arp.table_get_entry(it);
        if (! result)
            continue;
        std::cout << '\t' << Color::INET << result.value().ipv4 << Color::NONE
                  << '\t' << Color::MAC << result.value().mac << Color::NONE
                  << '\n';
    }
    std::cout << std::flush;
}

void
table(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    if (i == e)
        throw std::runtime_error{"Expected { show | add | delete"
                                 " | find | set | flush }"};

    if (is_prefix(*i, "show"))
        table_show(++i, e, arp);
    else if (is_prefix(*i, "add"))
        table_add(++i, e, arp);
    else if (is_prefix(*i, "delete"))
        table_delete(++i, e, arp);
    else if (is_prefix(*i, "find"))
        table_find(++i, e, arp);
    else if (is_prefix(*i, "set"))
    {
        if (++i == e)
            throw std::runtime_error{"Expected lifetime"};

        if (is_prefix(*i, "lifetime"))
            table_set_lifetime(++i, e, arp);
        else
        {
            std::ostringstream error;
            error << "Unknown ACTION '" << *i
                  << "', try '" BIN_NAME " arp help'";
            throw std::runtime_error{error.str()};
        }
    }
    else if (is_prefix(*i, "flush"))
        table_flush(++i, e, arp);
    else
    {
        std::ostringstream error;
        error <<  "Unknown ACTION '" << *i << "', try '" BIN_NAME " arp help'";
        throw std::runtime_error{error.str()};
    }
}

void
show(CmdLineIterator i, CmdLineIterator e, na::arp & arp)
{
    std::cout << "server: ";
    if (arp.is_server_enabled())
        std::cout << Color::CONFIGURED << "enabled" << Color::NONE;
    else
        std::cout << Color::UNCONFIGURED << "disabled" << Color::NONE;

    std::cout << "\nreply_timeout: " << arp.get_client_timeout().value()
              << std::endl;

    table_show(i, e, arp);
}

void
parse_scope(CmdLineIterator i, CmdLineIterator e,
            na::arp & arp)
{
    if (i == e || is_prefix(*i, "help"))
        std::cout << USAGE << std::flush;
    else if (is_prefix(*i, "show"))
        show(++i, e, arp);
    else if (is_prefix(*i, "set"))
        set(++i, e, arp);
    else if (is_prefix(*i, "table"))
        table(++i, e, arp);
    else
    {
        std::ostringstream error;
        error <<  "Unknown ACTION '" << *i << "', try '" BIN_NAME " arp help'";
        throw std::runtime_error{error.str()};
    }
}

} // namespace

void
run(CmdLineIterator i, CmdLineIterator e,
    hw::core & root,
    Application::StackType stack_type,
    std::uint32_t stack_index)
{
    na::arp arp{create_arp(root, stack_type, stack_index)};
    parse_scope(i, e, arp);
}

} // namespace ArpScope

ENYX_CORES_NAMESPACE_END
