#include <boost/program_options.hpp>

#include <fstream>
#include <iostream>
#include <iterator>

#include <stdio.h>

#include <enyx/cores/i2c/i2c.hpp>

#include <enyx/hw/accelerator.hpp>
#include <enyx/hw/mmio.hpp>
#include <enyx/hw/core_tree.hpp>
#include <enyx/hw/core.hpp>

#include "../lib/DeviceDiscovery.hpp"

namespace po = boost::program_options;

enum class i2c_command {
    READ,
    WRITE,
};

struct run_params {
    i2c_command command;
    std::string accelerator;
    uint8_t bus_id = 0;
    uint8_t size = 0;
    uint8_t addr = 0x0;
    bool reg_op = false;
    uint8_t reg_addr = 0x0;
    uint8_t data = 0x0;
    bool ascii = false;
};

void
run_i2c_command(run_params const & params)
{
    auto accelerator = enyx::create_accelerator(params.accelerator);
    auto mmio = enyx::get_first_mmio(accelerator);
    auto core_tree = enyx::hw::enumerate_cores(mmio);
    auto root = core_tree.get_root();
    auto i2c = enyx::i2c::i2c(root);
    auto bus = i2c.get_bus(params.bus_id).v();

    bus.set_clock_prescale(ENYX_I2C_CLOCK_PRESCALE_100KHz);
    bus.set_read_timeout(2000);

    if (params.command == i2c_command::READ) {
        std::vector<uint8_t> data(params.size);
        size_t read_size;
        if (params.reg_op) {
            read_size = bus.read_register(params.addr, params.reg_addr,
                                          data.data(), params.size).v();
        } else {
            read_size = bus.read(params.addr, data.data(), params.size).v();
        }

        std::cout << "Read ";
        if (params.ascii)
            std::cout << std::endl;

        for (size_t i = 0; i < read_size; i++)
            if (params.ascii)
                std::cout << data[i];
            else
                std::cout << "0x" << std::hex << (unsigned) data[i] << " ";
        std::cout << std::endl;

    } else {
        if (params.reg_op) {
            bus.write_register(params.addr, params.reg_addr,
                               &params.data, 1).v();
        } else {
            bus.write(params.addr, &params.data, 1).v();
        }
        std::cout << "Successfully wrote data" << std::endl;
    }
}

run_params
handle_options(po::variables_map args)
{
    run_params params;

    if (! args.count("accelerator")) {
        throw std::runtime_error("--accelerator option is mandatory");
    }
    params.accelerator = args["accelerator"].as<std::string>();

    if (! args.count("bus")) {
        throw std::runtime_error("--bus option is mandatory");
    }
    params.bus_id = args["bus"].as<uint32_t>();

    if (args.count("read") == args.count("write")) {
        throw std::runtime_error("Either --read or --write must be specified");
    }

    if (! args.count("size")) {
        throw std::runtime_error("--size must be specified");
    }
    params.size = args["size"].as<std::uint32_t>();

    if (! args.count("addr")) {
        throw std::runtime_error("--addr must be specified");
    }
    params.addr = std::stoul(args["addr"].as<std::string>(), nullptr, 16);

    if (args.count("reg-addr")) {
        params.reg_op = true;
        params.reg_addr =
            std::stoul(args["reg-addr"].as<std::string>(), nullptr, 16);
    }

    if (args.count("ascii"))
        params.ascii = true;

    if (args.count("read")) {
        std::cout << "Reading "
            << (unsigned) params.size << " bytes "
            << "from device at address 0x"
            << std::hex << (unsigned) params.addr << " ";
        if (params.reg_addr) {
            std::cout << "from register 0x"
            << std::hex << (unsigned) params.reg_addr;
        }
        std::cout << std::endl;

        params.command = i2c_command::READ;
    }

    if (args.count("write")) {
        if (params.size != 1) {
            throw std::runtime_error("only size 1 writes are supported");
        }

        if (! args.count("data")) {
            throw std::runtime_error("--data must be specified with --write");
        }
        params.data = std::stoul(args["data"].as<std::string>(), nullptr, 16);

        std::cout << "Writing "
            << (unsigned) params.size << " bytes "
            << "to device at address 0x"
            << std::hex << (unsigned) params.addr << " ";
        if (params.reg_addr) {
            std::cout << "to register 0x"
            << std::hex << (unsigned) params.reg_addr;
        }
        std::cout << std::endl;

        params.command = i2c_command::WRITE;
    }

    return params;
}

po::variables_map
parse(int ac, char* av[])
{
    po::options_description desc("Allowed options");
    desc.add_options()
        ("help,h", "produce help message")
        ("accelerator,a",
         po::value<std::string>(), "accelerator index or name")
        ("bus,b", po::value<uint32_t>(), "i2c bus index")
        ("read,r", "read")
        ("write,w", "write")
        ("addr", po::value<std::string>(), "device address in hexadecimal notation")
        ("reg-addr", po::value<std::string>(), "register address in hexadecimal notation")
        ("data,d", po::value<std::string>(), "data byte to read/write in hexadecimal notation")
        ("size,s", po::value<uint32_t>(), "size to read/write in bytes")
        ("ascii,c", "read in ascii")
        ;

    po::variables_map args;

    po::store(po::parse_command_line(ac, av, desc), args);
    po::notify(args);

    if (args.count("help")) {
        std::cout << desc << std::endl;
        exit(0);
    }

    return args;
}

int main(int ac, char* av[])
{
    auto args = parse(ac, av);

    auto params = handle_options(args);
    run_i2c_command(params);

    return 0;
}

