/// @file header.h
#pragma once

#include <assert.h>
#include <stdint.h>

/// The version of the UDP header protocol
#define ENYX_UDP_HEADER_VERSION_1 (1U)

/// The mask of the UDP reset flag
#define ENYX_UDP_HEADER_FLAG_RESET (1U << 2)

/**
 *  This structure represents the header used to receive from the UDP TOE
 *
 *  Is used internally by the UDP IO Source.
 *
 *  A user can use it directly if he can't leverage on the source
 *  API for some reason.
 */
typedef struct {
    /// The byte size of this header + payload
    uint16_t size;

    /// The version of the header
    uint8_t version;

    /// The flag associated with the current message
    uint8_t flags;

    /// The UDP session id concerned by the current message
    uint32_t session_id;

    /// IP address of the sender
    uint32_t remote_ip;

    /// Empty field
    uint16_t empty;

    /// Port of the sender
    uint16_t remote_port;
} enyx_udp_source_header;

/**
 *  This structure represents the header used to send through UDP TOE
 *
 *  Is used internally by the UDP IO Sink.
 *
 *  A user can use it directly if he can't leverage on the sink
 *  API for some reason.
 */
typedef struct {
    /// The byte size of this header + payload
    uint16_t size;

    /// The version of the header
    uint8_t version;

    /// The flag associated with the current message
    uint8_t flags;

    /// The UDP session id concerned by the current message
    uint32_t session_id;
} enyx_udp_sink_header;

/// @cond
_Static_assert(sizeof(enyx_udp_source_header) == 16,
               "udp source header is 16 bytes long");
/// @endcond

/// @cond
_Static_assert(sizeof(enyx_udp_sink_header) == 8,
               "udp sink header is 8 bytes long");
/// @endcond
