#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/hw_c/core.h>

/**
 * The main UART subsystem struct.
 */
typedef struct enyx_uart enyx_uart;

/**
 * Get the UART core clock frequency.
 *
 * @param uart A pointer to the uart struct.
 * @param frequency The UART core clock frequency
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_uart_get_clock_frequency(enyx_uart const * uart, uint32_t * frequency);

/**
 * Get the UART baudrate.
 *
 * @param uart A pointer to the uart struct.
 * @param baudrate The UART baudrate
 * @return 0 on success, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_uart_get_baudrate(enyx_uart const * uart, uint32_t * baudrate);

/**
 * Check if the UART core RX fifo is empty.
 *
 * @param uart A pointer to the uart struct.
 * @param empty ``true`` if the fifo is empty, ``false`` otherwise
 * @return 0 on success, -1 on failure with @b errno set accordingly.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_uart_is_rx_empty(enyx_uart const * uart, bool * empty);

/**
 * Check if the UART core TX fifo is full.
 *
 * @param uart A pointer to the uart struct.
 * @param full ``true`` if the fifo is full, ``false`` otherwise
 * @return 0 on success, -1 on failure with @b errno set accordingly.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_uart_is_tx_full(enyx_uart const * uart, bool * full);

/**
 * Check if the UART RX lane is busy (data transfer in progress).
 *
 * @param uart A pointer to the uart struct.
 * @param busy ``true`` if the lane is busy, ``false`` otherwise
 * @return 0 on success, -1 on failure with @b errno set accordingly.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_uart_is_rx_busy(enyx_uart const * uart, bool * busy);

/**
 * Check if the UART TX lane is busy (data transfer in progress).
 *
 * @param uart A pointer to the uart struct.
 * @param busy ``true`` if the lane is busy, ``false`` otherwise
 * @return 0 on success, -1 on failure with @b errno set accordingly.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_uart_is_tx_busy(enyx_uart const * uart, bool * busy);

/**
 * Set the UART baudrate divisor.
 *
 * @param uart A pointer to the uart struct.
 * @param divisor The baudrate divisor to apply.
 * @return 0 sucess, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_uart_set_baudrate_divisor(enyx_uart * uart, uint16_t divisor);

/**
 * Clear the UART core RX fifo.
 *
 * @param uart A pointer to the uart struct.
 * @return 0 sucess, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_uart_clear_rx_fifo(enyx_uart * uart);

/**
 * Clear the UART core TX fifo.
 *
 * @param uart A pointer to the uart struct.
 * @return 0 sucess, -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
int
enyx_uart_clear_tx_fifo(enyx_uart * uart);

/**
 * Read data from the UART core RX fifo.
 *
 * Read data until the RX fifo is empty, or the requested size is reached,
 * without waiting for new data.
 *
 * @param uart A pointer to the uart struct.
 * @param data A pointer to read the data to.
 * @param size The size of the @p data pointer.
 * @return The amount of data read from the fifo on success, -1 on failure
 * (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
ssize_t
enyx_uart_read(enyx_uart * uart, uint8_t * data, size_t size);

/**
 * Try to read @p size data from the UART core RX fifo.
 *
 * Read data until the requested size is reached, or the given timeout is
 * reached when waiting for new data. On timeout, the returned read size will
 * be smaller than the requested size.
 *
 * @param uart A pointer to the uart struct.
 * @param data A pointer to read the data to.
 * @param size The size of the @p data pointer.
 * @param timeout_ms The maximum time to wait for new data.
 * @return The amount of data read on success, -1 on failure (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
ssize_t
enyx_uart_read_all(enyx_uart * uart, uint8_t * data, size_t size,
                   uint64_t timeout_ms);

/**
 * Write data to the UART core TX fifo.
 *
 * Write data until the TX fifo is full, or all the given data is written,
 * without waiting to try to send more data.
 *
 * @note If an error occurs during after part of the data has been successfuly written, the function will brea
 *
 * @param uart A pointer to the uart struct.
 * @param data A pointer to the data to write.
 * @param size The size of the @p data pointer.
 * @return The amount of data written to the fifo on success. On failure, @b
 * errno will be set. The function will return -1 if no data at all could be
 * sent or the partial amount of data sent if some data was sent.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
ssize_t
enyx_uart_write(enyx_uart * uart, const uint8_t * data, size_t size);

/**
 * Try to write @p size data to the UART core TX fifo.
 *
 * Try to write data until all given data is written, or the given timeout is
 * reached when waiting for fifo space. On timeout, the returned written size
 * will be smaller than the requested size.
 *
 * @param uart A pointer to the uart struct.
 * @param data A pointer to the data to write.
 * @param size The size of the @p data pointer.
 * @param timeout_ms The maximum time to wait for data to be sent.
 * @return The amount of data written to the fifo on success, -1 on failure
 * (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
ssize_t
enyx_uart_write_all(enyx_uart * uart, const uint8_t * data, size_t size,
                    uint64_t timeout_ms);

/**
 * Create an UART struct from a core subtree.
 *
 * Enumerates the subtree until a muart core is found and create an @b uart
 * struct from it.
 *
 * @param subtree The core subtree to look up.
 * @return The UART struct on success. NULL on failure (with @b errno set
 * accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
enyx_uart *
enyx_uart_create(enyx_hw_core * subtree);

/**
 * Destroy an UART struct.
 *
 * @param uart The UART struct to free.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY
void
enyx_uart_destroy(enyx_uart * uart);
