#pragma once

#include <stdbool.h>
#include <stdint.h>

#include <enyx/hw_c/core.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/types.h>

/**
 * TCP window size value used to set the window size value to 0 in
 * @ref enyx_tcp_session_parameters.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY uint16_t const ENYX_TCP_ZERO_WINDOW;

/**
 * Type representing a session.
 *
 * This can only be obtained from a tcp object.
 */
typedef struct {
    /// Session ID
    uint16_t session_id;
    /// TCP subsystem this session is associated with
    struct enyx_tcp * tcp;
} enyx_tcp_session;

/// Default Maximum Segment Size
static const uint16_t ENYX_TCP_DEFAULT_MSS = 1424;
/// Default window size
static const uint16_t ENYX_TCP_DEFAULT_WINDOW_SIZE = 40000;
/// Default retry timeout in millisecond
static const uint16_t ENYX_TCP_DEFAULT_RETRY_TIMEOUT = 10;

/**
 * TCP session open parameters.
 *
 * This is used both for client and server session opening.
 */
typedef struct {
    /// Interface to use when opening session.
    uint16_t interface_id;

    /// Source port to use for connection. Use 0 for random port.
    uint16_t source_port;

    /**
     * Maximum Segment Size in bytes. Use 0 for default value (@b
     * ENYX_TCP_DEFAULT_MSS).
     */
    uint16_t mss;

    /**
     * Window size in bytes. Use 0 for default value (@b
     * ENYX_TCP_DEFAULT_WINDOW_SIZE).
     * Use the @ref ENYX_TCP_ZERO_WINDOW constant value to set the window size
     * to 0.
     */
    uint16_t window_size;

    /// Window scale factor. Use 0 for default value (hardware default).
    uint8_t window_scale_factor;

    /**
     * Retry timeout in millisecond. Use 0 for default value (@b
     * ENYX_TCP_DEFAULT_RETRY_TIMEOUT)
     */
    uint16_t retry_timeout;

    /// Enable timestamp in TCP header.
    bool enable_timestamp;

    /// Enable instant ack.
    bool enable_instant_ack;

    /// Disable window shrinking.
    bool disable_window_shrinking;
} enyx_tcp_session_parameters;

/**
 * Connect to server @p peer_address @p port on @p session.
 *
 * @param session Session to connect.
 * @param peer_address Remote IPv4 address to connect to.
 * @param peer_port Remote port to connect to.
 * @param parameters Session parameters
 * @return 0 if session opening was successful. -1 on error (with @b errno set
 *         accordingly).
 *
 * @note This function does not wait for the sesssion to be established.
 *       Please check session status afterwards.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_connect(enyx_tcp_session * session,
                             enyx_ipv4_address const * peer_address,
                             uint16_t peer_port,
                             enyx_tcp_session_parameters const * parameters);

/**
 * Listen on session @p session. Port is specified via @p parameters port.
 *
 * @param session Session used for listening.
 * @param parameters Session parameters
 * @return 0 if session opening was successful. -1 on error (with @b errno set
 *         accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_listen(enyx_tcp_session * session,
                            enyx_tcp_session_parameters const * parameters);

/**
 * Close session @p session.
 *
 * @param session to close.
 * @return 0 if session was sucessfuly closed. -1 on error (with @b errno set
 *         accordingly).
 *
 * @note This function does not wait for the session to be actually closed.
 *       Please check the session status afterwards.
 *
 * @warning If the session was not closed after calling this function, it means
 * that the hardware was continuously transmitting data and could not close the
 * session in a reasonable time. It is advised to try again. If this keeps
 * failing use the @ref enyx_tcp_session_reset function to reset instead of
 * closing. Please check the hardware documentation for more details.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_close(enyx_tcp_session * session);

/**
 * Reset session @p session.
 *
 * @param session to reset.
 * @return 0 if session was sucessfuly reset. -1 on error (with @b errno set
 *         accordingly).
 *
 * @note This function does not wait for the session to be actually reset.
 *       Please check the session status afterwards.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_reset(enyx_tcp_session * session);

/**
 * Set the window size of an open @p session.
 *
 * @param session The session to change.
 * @param window_size The window size to set.
 * @return 0 on success. -1 On failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_set_window_size(enyx_tcp_session * session,
                                 uint16_t window_size);

/**
 * Get the window size of an open @p session.
 *
 * @param session An open session.
 * @param window_size The current window size.
 * @return 0 on success. -1 On failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_get_window_size(enyx_tcp_session const * session,
                                 uint16_t * window_size);

/**
 * Get the initial sequence number of an open @p session.
 *
 * @note The initial sequence number is dynamically generated according to
 *        RFC 793. See also the linux implementation in core/secure_seq.h
 *        for a similar implementation.
 *
 * @param session An open session.
 * @param init_seqnum The initial sequence number for this open session.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_get_init_seqnum(enyx_tcp_session const * session,
                                     uint32_t * init_seqnum);

/**
 * Get the source port used of an open @p session.
 *
 * @param session An open session.
 * @param source_port The source port.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_get_source_port(enyx_tcp_session const * session,
                                     uint16_t * source_port);

/**
 * A remote TCP endpoint.
 */
typedef enyx_ipv4_endpoint enyx_tcp_endpoint;


/**
 * Get the peer (remote) @p address and @p port of an established @p session.
 *
 * @param session An established session.
 * @param peer The peer IPv4 and Port address.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_getpeername(enyx_tcp_session const * session,
                                 enyx_tcp_endpoint * peer);

/**
 * Possible state of sessions.
 */
typedef enum {
    /// Closed session.
    ENYX_TCP_SESSION_STATE_CLOSED,
    /// Session is opening.
    ENYX_TCP_SESSION_STATE_OPENING,
    /// Session is established.
    ENYX_TCP_SESSION_STATE_ESTABLISHED,
    /// Session is about to be closed.
    ENYX_TCP_SESSION_STATE_CLOSING,
} enyx_tcp_session_state;

/**
 * Get the state of a session.
 *
 * @param session The session.
 * @param state The state output.
 * @return 0 on success. -1 on failure (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_get_state(enyx_tcp_session const * session,
                               enyx_tcp_session_state * state);

/**
 * Session status (as seen from hardware).
 */
typedef enum {
    /** represents no connection state at all. */
    ENYX_TCP_SESSION_DETAILED_STATUS_CLOSED = 0,

    /**
     * represents waiting for a connection request from any remote TCP and
     * port.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_LISTEN = 1,
    /**
     * represents waiting for a matching connection request after having
     * sent a connection request.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_SYN_SENT = 3,
    /**
     * represents waiting for a confirming connection request acknowledgment
     * after having both received and sent a connection request.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_SYN_RCVD = 2,
    /**
     * represents an open connection, data received can be delivered to the
     * user. The normal state for the data transfer phase of the connection.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_ESTABLISHED = 4,
    /**
     * represents waiting for a connection termination request from the
     * remote TCP, or an acknowledgment of the connection termination
     * request previously sent.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_FIN_WAIT1 = 7,
    /**
     * represents waiting for a connection termination request from the
     * remote TCP.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_FIN_WAIT2 = 8,
    /**
     * represents waiting for a connection termination request from the
     * local user.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_CLOSE_WAIT = 5,
    /**
     * represents waiting for a connection termination request
     * acknowledgment from the remote TCP.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_CLOSING = 9,
    /**
     * represents waiting for an acknowledgment of the connection
     * termination request previously sent to the remote TCP (which includes
     * an acknowledgment of its connection termination request).
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_LAST_ACK = 6,
    /**
     * represents waiting for enough time to pass to be sure the remote TCP
     * received the acknowledgment of its connection termination request.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_TIME_WAIT = 10,
    /**
     * represents waiting for enough time to pass to be sure the TCP stack
     * was reseted properly.
     */
    ENYX_TCP_SESSION_DETAILED_STATUS_RESET_WAIT = 11,
} enyx_tcp_session_detailed_status;

typedef enum {
    ENYX_TCP_SESSION_CONGESTION_CONTROL_STATE_SLOW_START = 0,
    ENYX_TCP_SESSION_CONGESTION_CONTROL_STATE_CONGESTION_AVOID = 1,
    ENYX_TCP_SESSION_CONGESTION_CONTROL_STATE_FAST_RETX = 2,
} enyx_tcp_session_congestion_control_state;

/**
 * A structure representing a session's runtime statistics.
 */
typedef struct {
    /// TCP retry memory mask
    uint32_t retry_memory_mask;
    /// TCP session detailed status
    enyx_tcp_session_detailed_status status;
    /// TCP Receive Next Sequence Number (RCV.NXT)
    uint32_t rx_next_sequence_number;
    /// TCP Last Ack Number Received
    uint32_t rx_last_ack;
    /// TCP Send Next Sequence Number (SND.NXT)
    uint32_t tx_next_sequence_number;
    /// TCP Last Ack Number Sent
    uint32_t tx_last_ack;
    /// TCP Out Of Sequence MAX
    uint32_t rx_out_of_sequence_max;
    /// TCP retransmission Current Sequence Number
    uint32_t retry_current_seqnum;
    /**
     * TCP retransmission Current Count for the sequence number @b
     * retry_current_seqnum
     */
    uint8_t retry_current_count;
    /// TCP retransmission Current Retransmit Date
    uint16_t retry_current_date;
    /// TCP retransmission Write Data Memory Pointer
    uint32_t retry_memory_write_pointer;
    /// TCP retransmission Read Data Memory Pointer
    uint32_t retry_memory_read_pointer;
    /// Transmission Split Size value in bytes
    uint16_t tx_split_size;
    /// Remote MSS read in the SYN/SYN-ACK options header
    uint16_t remote_mss;
    /// Remote Advertised window size
    uint16_t remote_window_size;
    /// Remote Scale Factor read in the SYN/SYN-ACK options header
    uint8_t remote_window_scale_factor;
    /// Timestamp Enable Indicator read in the SYN/SYN-ACK options header
    bool remote_timestamp_enabled;
    /// Last timestamp value received
    uint32_t remote_timestamp_value;
    /// Number of segments received
    uint32_t rx_segment_count;
    /// Number of dropped segments (due to slow user net_interface)
    uint32_t rx_dropped_segment_count;
    /// Number of Out of Sequence segments (due to missing packets)
    uint32_t rx_out_of_sequence_segment_count;
    /// Number of user packets split because of remote MSS value
    uint32_t tx_split_count;
    /// Number of Tx packets
    uint32_t tx_state_management_packet_count;
    /// Number of ACK only packets
    uint32_t tx_ack_only_count;
    /// Number of discared RX packets
    uint32_t rx_discarded_count;
    /// Number of segments that have been retransmitted
    uint32_t retry_count;
    /**
     * Number of user packets received coming from the normal data path (as
     * opposed to the re-ordering module)
     */
    uint32_t rx_user_packet_count;
    /// Number of user bytes received
    uint32_t rx_user_byte_count;
    /// Number of user packets transmitted
    uint32_t tx_user_packet_count;
    /// Number of user bytes transmitted
    uint32_t tx_user_byte_count;
    /// Number of user packets dropped
    uint32_t tx_user_drop_count;
    /// Number of user packets received coming from the re-ordering module
    uint32_t rx_user_reordering_packet_count;
    /// Indicates that the fast retransmit mechanism has triggered
    bool congestion_control_send_fast_retransmit;
    /// Number of dupplicated ACK received
    uint8_t congestion_control_dupplicate_ack_count;
    /// Congestion Control Current state
    enyx_tcp_session_congestion_control_state congestion_control_state;
    /// Current congestion window
    uint32_t congestion_window;
    /// Threshold to drive transition from slow start to congestion avoid
    uint32_t congestion_threshold;
    /// Seqnum to recover to leave fast retransmission mechanism
    uint32_t congestion_recover_seqnum;
    /// Last credit count given to the user
    uint32_t credit_event_sent;
    /// Counter of user TX byte when last event was sent
    uint32_t tx_user_byte_count_sent;
} enyx_tcp_session_statistics;

/// @cond
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_tcp_session_get_statistics_(enyx_tcp_session const * session,
                                     enyx_tcp_session_statistics * statistics,
                                     size_t statistics_size);
/// @endcond

/**
 * Get the TCP subsystem session Hardware Statistics
 *
 * @param tcp The tcp subsystem
 * @param statistics The session statistics object
 *
 * @return 0 on success, -1 on failure (with @b errno set accordingly)
 */
#define enyx_tcp_session_get_statistics(tcp, statistics) \
    enyx_tcp_session_get_statistics_(tcp, statistics, sizeof(*statistics))
