/**
 * @file
 *
 * Contains the types and functions related to the collector.
 */
#pragma once

#include <stdint.h>

#include <enyx/hw_c/mmio.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/probes/collector.h>
#include <enyx/cores_c/probes/mtg.h>

/**
 * Represents a system.
 *
 * This class can be used to retrieve collectors and their associated probes.
 * The system associates probes and collectors based on `COLLECTOR_NAME`
 * string register of probe and collector cores and not on actual bus
 * connections.
 *
 * This API is legacy and its use is discouraged. Users are encouraged to use
 * probes and collectors separately and associate them manually based on how
 * those are internally connected.
 *
 * They contain one @ref enyx_probes_mtg, one or more
 * @ref enyx_probes_collector and their associated @ref enyx_probes_probe.
 *
 * @since 6.7.0
 */
typedef struct enyx_probes_system enyx_probes_system;

/**
 * Create a system from a core found under @p root core.
 *
 * @param root The root core from which to search for event system.
 * @return The requested system on success, @b NULL on error with errno set.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_probes_system *
enyx_probes_system_create(enyx_hw_core const * root);

/**
 * Destroy a system.
 *
 * @param sys system to destroy.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_probes_system_destroy(enyx_probes_system * sys);

/**
 * Enumerate event collectors.
 *
 * @param sys The system to use.
 * @param collectors The array to fill with the retrieved event collectors.
 * @param capacity The array capacity.
 * @return The count of retrieved collectors on success,
 *         -1 with @b errno set on failure.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_probes_system_enumerate_collectors(enyx_probes_system const* sys,
                                        enyx_probes_collector ** collectors,
                                        size_t capacity);

/**
 * Retrieve the mtg belonging to a system.
 *
 * @param sys The sys to search for mtg.
 * @return The mtg on success, @b NULL with errno set on failure.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_probes_mtg *
enyx_probes_system_get_mtg(enyx_probes_system const* sys);

/**
 * Enumerate the related probes.
 *
 * @param system The system to use.
 * @param collector The collector to query.
 * @param probes An array filled with the probes instances.
 * @param capacity The array capacity.
 * @return The discovered probes count or -1 with @b errno set on failure.
 *
 * @note The discovered probes count can be greater than @p capacity.
 *       But only up to @p capacity probes handles are written to @p probes.
 *       The caller should detect the truncation by comparing the return value
 *       with the @p probes buffer @p capacity.
 *
 * @since 6.7.0
 */
ENYX_CORES_C_SYMBOL_VISIBILITY ssize_t
enyx_probes_system_enumerate_probes(enyx_probes_system * system,
                                    enyx_probes_collector const * collector,
                                    enyx_probes_probe ** probes,
                                    size_t capacity);
