/// @file igmp.h
#pragma once

#include <stdbool.h>

#include <enyx/hw_c/core.h>

#include <enyx/cores_c/symbol_visibility.h>

/**
 * @brief IGMP subsystem object.
 *
 * This object represents the IGMP subsystem of a hardware network stack.
 *
 * Usage:
 * @code{.c}
 * int result = -1;
 * enyx_igmp * igmp;
 *
 * // Create the IGMP object
 * igmp = enyx_igmp_create(udp_ip_10g_ull_stack_core);
 * if (! igmp)
 * {
 *     perror("enyx_igmp_create");
 *     goto create_failed;
 * }
 *
 * // Get and print the current configuration of the IGMP subsystem
 * printf("IGMP is %s\n",
 *        enyx_igmp_enabled(igmp) ? "enabled" : "disabled");
 *
 * // Disable IGMP
 * if (enyx_igmp_disable(igmp) < 0)
 * {
 *     perror("enyx_igmp_disable");
 *     goto disable_failed:
 * }
 *
 * result = 0;
 * disable_failed:
 * enyx_igmp_destroy(igmp);
 * create_failed:
 * return result;
 * @endcode
 */
typedef struct enyx_igmp enyx_igmp;

/**
 * @brief Construct the IGMP subsystem.
 *
 * This module configures the following hw core:
 *
 * * nxudp_ip_10g_ull (from UDP_ULL 3.x)
 *
 * @param stack The hardware core containing the stack.
 * @note This requires you to find a compatible hw::core core using the
 *       enyx-hw library (e.g. using enyx_hw_core_enumerate).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY enyx_igmp *
enyx_igmp_create(enyx_hw_core * stack);

/**
 * @brief Destroy an IGMP subsystem.
 *
 * @param igmp The IGMP subsystem to destroy.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY void
enyx_igmp_destroy(enyx_igmp * igmp);

/**
 * @brief Check if IGMP is enabled.
 *
 * @param igmp The IGMP subsystem.
 * @return true if IGMP is enabled.
 */
ENYX_CORES_C_SYMBOL_VISIBILITY bool
enyx_igmp_enabled(enyx_igmp const * igmp);

/**
 * @brief Enable IGMP.
 *
 * @param igmp The subsystem to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_igmp_enable(enyx_igmp * igmp);

/**
 * @brief Disable IGMP.
 *
 * @param igmp The subsystem to use.
 * @return 0 on success, -1 on error (with @b errno set accordingly).
 */
ENYX_CORES_C_SYMBOL_VISIBILITY int
enyx_igmp_disable(enyx_igmp * igmp);
