/// @file source.h

#pragma once

#include <assert.h>
#include <stdint.h>

#include <enyx/cores_c/symbol_visibility.h>
#include <enyx/cores_c/data_stream/metadata.h>

/**
 *  This function pointer is called on each data received
 *
 *  The @p data is a pointer to a @p size bytes-long
 *  payload, with its related @p metadata.
 *
 *  The @p opaque pointer is equal to opaque
 *  argument provided to the enyx_data_stream_source_poll_once().
 *
 *  Example of conforming handler:
 *  @code
 *  static void
 *  on_data(uint8_t const * data,
 *          uint32_t size,
 *          void const  * metadata,
 *          void * opaque)
 *  {
 *      printf("Received %" PRIu32 " bytes\n", size);
 *  }
 *  @endcode
 */
typedef void
(*enyx_data_stream_on_data)(uint8_t const * data,
                            uint32_t size,
                            void const  * metadata,
                            void * opaque);

/**
 *  This type represents an IO source.
 *
 *  The purpose of an IO source is to provide data and metadata
 *  by invoking a user-provided callback.
 */
struct enyx_data_stream_source
{
    /**
     *  Optional pointer to the source implementation
     *
     *  This is useful if the source needs stateful data
     *  in order to perform its task, for example the decryption
     *  key if the source decrypts payload.
     */
    void * opaque;

    /**
     * Pointer to the next function to call when input data has been processed.
     *
     * This function should always be called with properly intialized metadata
     * pointer, with the error field set.
     */
    enyx_data_stream_on_data on_data;
};

/// Alias shortcut
typedef struct enyx_data_stream_source enyx_data_stream_source;
