/// @file metadata.h
#pragma once

#include <stdbool.h>

/**
 *  This type represents a metadata header used to create a linked list
 *  of metadata.
 *
 *  Depending on the sink/source type, metadata convey different informations.
 *  For example:
 *  - TCP sink/source: the TCP session id
 *  - UDP sink: the session id, the dest IP & dest port
 *  - UDP source: the session id, the src IP & src port
 *
 *  When receiving data, the user must read the associated metadata to
 *  retrieve the informations parsed by each source forming a chain.
 *  For example with a @b HW->TCP->HTTP source chain, the user is expected to
 *  get the HTTP Method & URL from the first metadata, and retrieve the
 *  second metadata (TCP) from it, which in turns contains the TCP session id.
 *
 *  The error state must be checked first to check if the provided data is
 *  expected to be valid.
 *  As each specific metadata type should start with this generic metadata
 *  structure, the error state should be available in every metadata type.
 *
 *  @code
 *  void const on_data(uint8_t const * data,
 *                     uint32_t data_size,
 *                     void const  * metadata,
 *                     void * opaque)
 *  {
 *      enyx_http_metadata const * http_metadata = metadata;
 *      if (http_metadata.metadata.error) {
 *          printf("Cannot decode HTTP data, provided data is invalid.\n");
 *          return;
 *      }
 *
 *      print("method %s, url %s\n",
 *            http_metadata->method, http_metadata->url);
 *
 *      enyx_tcp_data_stream_metadata const * tcp_metadata = http_metadata->metadata.next_metadata;
 *      print("tcp session id %" PRIu32 "\n",
 *            tcp_metadata->session_id);
 *  }
 *  @endcode
 *
 *  When sending data, the user must create each metadata required by the
 *  sinks populated in the sink chain he is using, and link them together.
 *
 *  @code
 *  enyx_tcp_data_stream_metadata const tcp_metadata = {
 *      .session_id = 42,
 *  };
 *  enyx_http_metadata const http_metadata = {
 *      .metadata.next_metadata = &tcp_metadata,
 *      .method = "PUT",
 *      .url = "www.example.com",
 *  };
 *  enyx_data_stream_sink_send(sink, "XXX", 3, &http_metadata);
 *  @endcode
 *
 *  The user can also create a custom metadata type specific to its source
 *  or sink with an attribute of type @c enyx_data_stream_metadata and use it to
 *  reference the previous source metadata or next sink metadata.
 *
 *  Usage:
 *  @code
 *  struct source_x_metadata
 *  {
 *      struct enyx_data_stream_metadata header;
 *      int value_useful_to_source_x;
 *  };
 *  @endcode
 *
 *  The user source or sink is then responsible of providing/reading
 *  the specific metadata, and linking it with previous layer metadata
 *  using the metadata header.
 */
struct enyx_data_stream_metadata
{
    /// Optional pointer to the next metadata
    struct enyx_data_stream_metadata const * next_metadata;

    /**
     * Field indicating if previous processing ended with an error.
     * Data is likely to be invalid when set to true.
     */
    bool error;
};

/// Alias shortchut
typedef struct enyx_data_stream_metadata enyx_data_stream_metadata;
