#pragma once
#include <utility>

#include <enyx/cores/types.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>
#include <enyx/cores_c/tcp/session.h>

ENYX_CORES_NAMESPACE_BEGIN
namespace tcp {

/// @copydoc enyx_tcp_session_state
enum class session_state {
    CLOSED = ENYX_TCP_SESSION_STATE_CLOSED,
    OPENING = ENYX_TCP_SESSION_STATE_OPENING,
    ESTABLISHED = ENYX_TCP_SESSION_STATE_ESTABLISHED,
    CLOSING = ENYX_TCP_SESSION_STATE_CLOSING,
};

/**
 * Print a TCP session @p state.
 *
 * @param out The stream to use
 * @param state The state to output
 * @return A reference to the output stream
 */
std::ostream &
operator<<(std::ostream & out, session_state const & state);

/// @copydoc enyx_tcp_endpoint
using endpoint = ipv4_endpoint;

/// @copydoc enyx_tcp_session_parameters
using session_parameters = ::enyx_tcp_session_parameters;

/// @copydoc enyx_tcp_session_statistics
using session_statistics = ::enyx_tcp_session_statistics;

/// @copydoc ENYX_TCP_ZERO_WINDOW
static auto const ZERO_WINDOW = ::ENYX_TCP_ZERO_WINDOW;

/**
 * @copydoc enyx_tcp_session
 */
class session
{
public:
    /**
     * Construct a session from a C session.
     *
     * @param tcp The C pointer to the tcp owner of the session
     * @param session The C session
     *
     * @note This constructor should not be used directly, sessions should be
     * retrieved from @ref tcp get_session or get_available_session instead.
     */
    session(std::shared_ptr<::enyx_tcp> const & tcp,
            ::enyx_tcp_session const & session);

    /**
     * Get the session id.
     *
     * @return The session id
     */
    std::uint16_t
    id() const noexcept;

    /**
     * Connect to a server @p peer_address @p port.
     *
     * @param peer_address Remote IPv4 address to connect to (in Host Byte
     * Order)
     * @param peer_port Remote port to connect to
     * @param parameters Session parameters
     * @return A result object containing an error on failure
     *
     * @note This function does not wait for the sesssion to be established.
     *       Please check session status afterwards.
     */
    result<void>
    connect(ipv4_address const & peer_address,
            std::uint16_t peer_port,
            session_parameters const & parameters) noexcept;

    /**
     * Listen on this session. Port is specified via @p parameters port.
     *
     * @param parameters Session parameters
     * @return A result object containing an error on failure
     */
    result<void>
    listen(session_parameters const & parameters) noexcept;

    /**
     * Close this session.
     *
     * @return A result object containing an error on failure
     *
     * @note This function does not wait for the session to be actually closed.
     * Please check the session status afterwards.
     *
     * @warning If the session was not closed after calling this function, it
     * means that the hardware was continuously transmitting data and could not
     * close the session in a reasonable time. It is advised to try again. If
     * this keeps failing use the @ref session::reset function to reset
     * instead of closing. Please check the hardware documentation for more
     * details.
     */
    result<void>
    close() noexcept;

    /**
     * Reset this session.
     *
     * @return A result object containing an error on failure
     *
     * @note This function does not wait for the session to be actually reset.
     *       Please check the session status afterwards.
     */
    result<void>
    reset() noexcept;

    /**
     * Set the window size of this session.
     *
     * @param window_size The window size to set
     * @return A result object containing an error on failure
     */
    result<void>
    set_window_size(std::uint16_t window_size) noexcept;

    /**
     * Get the window size of this session.
     *
     * @return A result object containing the current window size on success or
     * an error on failure
     */
    result<std::uint16_t>
    get_window_size() const noexcept;

    /**
     * Get the initial sequence number of this session.
     *
     * @note The initial sequence number is dynamically generated according to
     *        RFC 793. See also the linux implementation in
     *        core/secure_seq.h for a similar implementation.
     *
     * @param[out] init_seqnum
     * @return A result object containing the initial sequence number on
     *         on success or an error on failure
     */
    result<std::uint32_t>
    get_init_seqnum() const noexcept;

    /**
     * Get the source port used of this session.
     *
     * @return A result object containing the source port on on success or an
     *         error on failure
     */
    result<std::uint16_t>
    get_source_port() const noexcept;

    /**
     * Get the peer (remote) address and port of this session.
     *
     * @param[out] peer_address The peer IPv4 address (in Host Byte Order)
     * @param[out] peer_port The peer port
     * @return A result object containing a pair<peer_address, peer_port> on
     *         success or an error on failure
     */
    result<endpoint>
    get_peername() const noexcept;

    /**
     * @copybrief enyx_tcp_session_get_state
     *
     * @return A result object containing the session state or an error
     */
    result<session_state>
    get_state() const noexcept;


    /**
     * @copybrief enyx_tcp_session_get_statistics
     */
    result<session_statistics>
    get_statistics() const noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle
     */
    ::enyx_tcp_session *
    handle() noexcept;

    /**
     * Access to the C handle.
     *
     * @return the C handle
     */
    ::enyx_tcp_session const *
    handle() const noexcept;

private:
    std::shared_ptr<::enyx_tcp> tcp_;
    ::enyx_tcp_session session_;
};

} /* namespace tcp */
ENYX_CORES_NAMESPACE_END

#include <enyx/cores/tcp/session.ipp>
