#pragma once

#include <enyx/cores/data_stream/source.hpp>
#include "collector.hpp"

/// @cond
namespace std {

template<>
struct default_delete<::enyx_probes_event_source>
{
    void
    operator()(::enyx_probes_event_source * ptr) const
    {
        ::enyx_probes_event_source_destroy(ptr);
    }
};

} // namespace std
/// @endcond

ENYX_CORES_NAMESPACE_BEGIN

namespace probes {

/// @cond
class abstract_wrapper
{
public:
    virtual
    ~abstract_wrapper() = default;

    static void
    on_event(::enyx_probes_probe_id probe_id,
             ::enyx_probes_probe_event const * event,
             void * opaque)
    {
        auto thiz = reinterpret_cast<abstract_wrapper *>(opaque);
        thiz->call_on_event(probe_id, *event);
    }

    static void
    on_error(void * opaque)
    {
        auto thiz = reinterpret_cast<abstract_wrapper *>(opaque);
        thiz->call_on_error();
    }

private:
    virtual void
    call_on_event(std::uint8_t probe_id,
                  event const & event) = 0;

    virtual void
    call_on_error() = 0;
};

template<typename OnEvent, typename OnError>
class wrapper : public abstract_wrapper
{
public:
    wrapper(OnEvent && on_event,
            OnError && on_error)
        : on_event_(std::forward<OnEvent>(on_event))
        , on_error_(std::forward<OnError>(on_error))
    { }

private:
    virtual void
    call_on_event(std::uint8_t probe_id,
                  event const & event) override
    {
        on_event_(probe_id, event);
    }

    virtual void
    call_on_error() override
    {
        on_error_();
    }

    OnEvent on_event_;
    OnError on_error_;
};
/// @endcond

/**
 * C++ probe event source abstraction.
 *
 * It is used to process probe events from a collector DMA stream.
 *
 * Usage:
 * @code
 * auto on_event = [] (std::uint8_t probe_id, enyx::probes::event const& event) {
 *     std::cout << "Event received from probe " << (unsigned) probe_id << std::endl;
 * };
 *
 * auto on_error = [] () {
 *     throw std::runtime_error{"Received an error on collector DMA channel."};
 * };
 *
 * enyx::probes::source event_source{on_event, on_error};
 * enyx::data_stream::hw_source hw_source{collector_stream, event_source};
 *
 * while (! exit_requested)
 *     hw_source.poll_once();
 * @endcode
 */
class source
{
public:
    /**
     * Construct a probes event source.
     *
     * @warning
     * If the callbacks passed to the source constructor are neither
     * copy-able nor r-value references, the source object will *not* take
     * ownership of the callbacks. They should be explicitly moved if their
     * lifetime is shorter than the source object lifetime.
     *
     * @tparam OnEvent Type of the on_event callback
     * @param on_event The callback to invoke on probe event
     * @tparam OnError Type of the on_error callback
     * @param on_error The callback to invoke on erroneous event
     */
    template<typename OnEvent, typename OnError>
    source(OnEvent && on_event, OnError && on_error);

    /**
     * Probe event source callback to be invoked when probe event occurs.
     *
     * @param data The event data
     * @param size Data size
     * @param metadata Metadata associated with data
     */
    void
    operator()(std::uint8_t const * data,
               std::uint32_t size,
               void const * metadata);

    /**
     * Get the current source as C source.
     *
     * @return A C source instance
     */
    ::enyx_data_stream_source
    as_next_source() const noexcept;

    /**
     * Retrieve the @b C handle associated with this event source.
     *
     * @return The C handle
     */
    ::enyx_probes_event_source *
    handle() const noexcept;

private:
    std::unique_ptr<abstract_wrapper> wrapper_;
    std::unique_ptr<::enyx_probes_event_source> source_;
};

} /* namespace probes */

ENYX_CORES_NAMESPACE_END

#include <enyx/cores/probes/data_stream.ipp>
