ENYX_CORES_NAMESPACE_BEGIN
namespace net_interface {

namespace {

inline ipv4_address
prefix_to_mask(std::uint8_t prefix)
{
    if (prefix > 32)
    {
        std::ostringstream error;
        error << "IPv4 prefix (" << prefix << ") can't be greater than 32";
        throw std::runtime_error{error.str()};
    }

    ipv4_address mask;
    auto i = mask.begin();

    for ( ; prefix > 8; prefix -= 8)
        *i++ = 0xff;

    *i = 0xff << (8 - prefix);

    return mask;
}

} // namespace

inline
ip_config::ip_config() noexcept
    : config_()
{ }

inline
ip_config::ip_config(::enyx_net_interface_ip_config const& c_config) noexcept
    : config_(c_config)
{ }

inline
ip_config::ip_config(ipv4_address const& ip,
                     ipv4_address const& mask) noexcept
{
    config_.ip = *static_cast<enyx_ipv4_address const*>(ip);
    config_.mask = *static_cast<enyx_ipv4_address const*>(mask);
}

inline
ip_config::ip_config(ipv4_address const& ip,
                     std::uint8_t prefix)
    : ip_config(ip, prefix_to_mask(prefix))
{ }

inline ipv4_address
ip_config::get_ip() const noexcept
{
    return config_.ip;
}

inline ipv4_address
ip_config::get_mask() const noexcept
{
    return config_.mask;
}

inline std::uint8_t
ip_config::get_prefix() const noexcept
{
    std::uint8_t prefix = 0;

    ipv4_address mask{config_.mask};
    auto i = mask.begin();

    for (; *i == 0xff; ++i)
        prefix += 8;

    for (; *i & 0x80; *i <<= 1)
        ++ prefix;

    return prefix;
}

inline
ip_config::operator ::enyx_net_interface_ip_config *() noexcept
{
    return &config_;
}

inline
ip_config::operator ::enyx_net_interface_ip_config const*() const noexcept
{
    return &config_;
}

inline std::ostream &
operator<<(std::ostream & out, ip_config const& ip_config)
{
    std::ostream::sentry sentry{out};

    if (sentry)
    {
        char buffer[64];
        int const ret = enyx_net_interfaces_print_ip_config(ip_config, buffer,
                sizeof(buffer));

        if (ret < 0)
            throw std::system_error{errno, std::generic_category(),
                                    "enyx_net_interfaces_print_ip_config()"};

        if (std::size_t(ret) >= sizeof(buffer))
            throw std::logic_error{"enyx_net_interfaces_print_ip_config()"};

        out << buffer;
    }

    return out;
}

inline std::ostream &
operator<<(std::ostream & out, vlan_pcp const& vlan_pcp)
{
    std::ostream::sentry sentry{out};

    if (sentry)
    {
        auto const c_vlan_pcp =
                static_cast<::enyx_net_interface_vlan_pcp>(vlan_pcp);

        auto string = enyx_net_interfaces_get_vlan_pcp_string(c_vlan_pcp);

        if (! string)
            throw std::system_error{errno, std::generic_category(),
                    "enyx_net_interfaces_get_vlan_pcp_string()"};

        out << string;
    }

    return out;
}

inline std::istream &
operator>>(std::istream & in, vlan_pcp & pcp)
{
    std::istream::sentry sentry{in};

    if (sentry)
    {
        std::string str;
        in >> str;

        ::enyx_net_interface_vlan_pcp c_pcp;
        if (enyx_net_interfaces_parse_vlan_pcp(str.c_str(), &c_pcp) < 0)
            throw std::system_error{errno, std::generic_category(),
                    "enyx_net_interfaces_parse_vlan_pcp()"};

        pcp = static_cast<vlan_pcp>(c_pcp);
    }

    return in;
}

inline
vlan_config::vlan_config() noexcept
{ }

inline
vlan_config::vlan_config(::enyx_net_interface_vlan_config const& c_config) noexcept
    : config_(c_config)
{ }

inline
vlan_config::vlan_config(id id, pcp pcp, dei dei) noexcept
{
    config_.vlan_id = id;
    config_.pcp = static_cast<::enyx_net_interface_vlan_pcp>(pcp);
    config_.dei = dei;
}

inline vlan_config::id
vlan_config::get_id() const noexcept
{
    return config_.vlan_id;
}

inline vlan_config::pcp
vlan_config::get_pcp() const noexcept
{
    return static_cast<pcp>(config_.pcp);
}

inline vlan_config::dei
vlan_config::get_dei() const noexcept
{
    return config_.dei;
}

inline
vlan_config::operator ::enyx_net_interface_vlan_config * () noexcept
{
    return &config_;
}

inline
vlan_config::operator ::enyx_net_interface_vlan_config const* () const noexcept
{
    return &config_;
}

inline bool
operator==(vlan_config const& a, vlan_config const& b)
{
    return a.get_id() == b.get_id() &&
            a.get_pcp() == b.get_pcp() &&
            a.get_dei() == b.get_dei();
}

inline bool
operator!=(vlan_config const& a, vlan_config const& b)
{
    return !(a == b);
}

inline std::ostream &
operator<<(std::ostream & out, vlan_config const& vlan)
{
    std::ostream::sentry sentry{out};

    if (sentry)
    {
        char buffer[64];
        int const ret = enyx_net_interfaces_print_vlan_config(vlan, buffer,
                sizeof(buffer));

        if (ret < 0)
            throw std::system_error{errno, std::generic_category(),
                                    "enyx_net_interfaces_print_vlan_config()"};
        if (std::size_t(ret) >= sizeof(buffer))
            throw std::logic_error{"enyx_net_interfaces_print_vlan_config()"};

        out << buffer;
    }

    return out;
}

inline
net_interfaces::net_interfaces(enyx::hw::core const& net_interfaces_core)
    : net_interfaces_core_(net_interfaces_core),
      handle_(::enyx_net_interfaces_create(net_interfaces_core_.handle()))

{
    if (! handle_)
        throw std::system_error{errno, std::generic_category(),
                                "Failed to create net_interfaces"};
}

inline enyx::hw::core
net_interfaces::get_core() const noexcept
{
    return net_interfaces_core_;
}

inline uint16_t
net_interfaces::count() const noexcept
{
    return ::enyx_net_interfaces_count(handle());
}

inline bool
net_interfaces::configured(std::uint16_t interface_id) const noexcept
{
    return ::enyx_net_interfaces_configured(handle(), interface_id);
}

inline result<void>
net_interfaces::configure(std::uint16_t interface_id) noexcept
{
    if (::enyx_net_interfaces_configure(handle(), interface_id) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline result<void>
net_interfaces::unconfigure(std::uint16_t interface_id) noexcept
{
    if (::enyx_net_interfaces_unconfigure(handle(), interface_id) != 0)
        return std::error_code{errno, std::generic_category()};
    return {};
}

inline result<mac_address>
net_interfaces::get_mac(std::uint16_t interface_id) const noexcept
{
    enyx_mac_address mac;
    if (::enyx_net_interfaces_get_mac(handle(), interface_id, &mac) != 0)
        return {std::error_code{errno, std::generic_category()}};
    return mac_address{mac};
}

inline result<void>
net_interfaces::set_mac(std::uint16_t interface_id,
                        mac_address const & mac) noexcept
{
    if (::enyx_net_interfaces_set_mac(handle(), interface_id, mac) != 0)
        return {std::error_code{errno, std::generic_category()}};
    return {};
}

inline result<ip_config>
net_interfaces::get_ip_config(std::uint16_t interface_id) const noexcept
{
    ::enyx_net_interface_ip_config address;
    if (::enyx_net_interfaces_get_ip_config(handle(), interface_id, &address) != 0)
        return {std::error_code{errno, std::generic_category()}};
    return address;
}

inline result<void>
net_interfaces::set_ip_config(std::uint16_t interface_id,
                              ip_config const & address) noexcept
{
    if (::enyx_net_interfaces_set_ip_config(handle(), interface_id, address) != 0)
        return {std::error_code{errno, std::generic_category()}};
    return {};
}

inline result<ipv4_address>
net_interfaces::get_gateway(std::uint16_t interface_id) const noexcept
{
    ipv4_address address;
    if (::enyx_net_interfaces_get_gateway(handle(), interface_id, address) != 0)
        return {std::error_code{errno, std::generic_category()}};
    return address;
}

inline result<void>
net_interfaces::set_gateway(std::uint16_t interface_id,
                            ipv4_address const & address) noexcept
{
    if (::enyx_net_interfaces_set_gateway(handle(), interface_id, address) != 0)
        return {std::error_code{errno, std::generic_category()}};
    return {};
}

inline result<vlan_config>
net_interfaces::get_vlan(std::uint16_t interface_id) const noexcept
{
    ::enyx_net_interface_vlan_config config;
    if (::enyx_net_interfaces_get_vlan_config(handle(), interface_id, &config) != 0)
        return {std::error_code{errno, std::generic_category()}};
    return vlan_config{config};
}

inline result<void>
net_interfaces::set_vlan(std::uint16_t interface_id,
                         vlan_config const & config) noexcept
{
    if (::enyx_net_interfaces_set_vlan_config(handle(), interface_id, config) != 0)
        return {std::error_code{errno, std::generic_category()}};
    return {};
}

inline result<std::uint16_t>
net_interfaces::get_mtu() const noexcept
{
    std::uint16_t mtu;
    if (::enyx_net_interfaces_get_mtu(handle(), &mtu) != 0)
        return std::error_code{errno, std::generic_category()};
    return mtu;
}

inline result<statistics>
net_interfaces::get_statistics() const noexcept
{
    statistics ret{};
    if (::enyx_net_interfaces_get_statistics(handle(), &ret) != 0)
        return std::error_code{errno, std::generic_category()};
    return ret;
}

inline enyx_net_interfaces *
net_interfaces::handle() const noexcept
{
    return handle_.get();
}

} /* namespace net_interface */
ENYX_CORES_NAMESPACE_END
