#pragma once

#include <functional>
#include <list>

#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

ENYX_CORES_NAMESPACE_BEGIN

namespace mocking {

/**
 * Implements a basic mocked UART device with RX and TX fifos.
 *
 * It also provides callbacks to know whenever the RX fifo is empty or the TX
 * fifo is full.
 *
 * @tparam UartMock The generated muart core mock to inherit from
 * @tparam fifo_capacity The RX/TX fifos capacity
 */
template<class UartMock, std::uint32_t fifo_capacity>
struct uart_tester : UartMock {
    uart_tester()
    {
        this->fifo_control_tx_data.on_write = [&] (std::uint64_t value)
            -> result<void> {
                tx_data_reg = value;
                return {};
        };
        this->fifo_control_tx_write.on_write = [&] (std::uint64_t value)
            -> result<void> {
                if (! (value & 0x01))
                    // Clear command
                    return {};

                if (tx_fifo.size() >= fifo_capacity)
                    throw std::system_error();

                tx_fifo.push_back(tx_data_reg);

                return {};
        };

        this->fifo_control_rx_read.on_write = [&] (std::uint64_t value)
            -> result<void> {
                if (! (value & 0x01))
                    // Clear command
                    return {};

                if (rx_fifo.empty())
                    throw std::system_error();

                rx_fifo.pop_front();
                return {};
        };
        this->fifo_control_rx_data.on_read = [&] ()
            -> result<uint64_t> {
                if (rx_fifo.empty())
                    throw std::system_error();

                return rx_fifo.front();
        };

        this->fifo_status_tx_full.on_read = [&] ()
            -> result<uint64_t> {
                auto full = tx_fifo.size() >= fifo_capacity;
                if (full && on_tx_full)
                    on_tx_full();
                return full;
        };

        this->fifo_status_rx_empty.on_read = [&] ()
            -> result<uint64_t> {
                auto empty = rx_fifo.empty();
                if (empty && on_rx_empty)
                    on_rx_empty();
                return empty;
        };

        this->fifo_control_rx_clear.on_write = [&] (std::uint64_t value)
            -> result<void> {
                if (! (value & 0x01))
                    // Write command
                    return {};

                rx_fifo.clear();
                return {};
        };

        this->fifo_control_tx_clear.on_write = [&] (std::uint64_t value)
            -> result<void> {
                if (! (value & 0x01))
                    // Write command
                    return {};

                tx_fifo.clear();
                return {};
        };
    };

    uint8_t tx_data_reg;
    std::list<uint8_t> rx_fifo;
    std::list<uint8_t> tx_fifo;

    std::function<void()> on_rx_empty;
    std::function<void()> on_tx_full;
};

}

ENYX_CORES_NAMESPACE_END
