#pragma once

#include <cstdint>
#include <memory>

#include <enyx/cores_c/memory_tester/memory_tester.h>

#include <enyx/hw/core.hpp>
#include <enyx/cores/macros.hpp>
#include <enyx/cores/namespace.hpp>
#include <enyx/cores/result.hpp>

ENYX_CORES_DEFAULT_DELETE(memory_tester);

ENYX_CORES_NAMESPACE_BEGIN

namespace memory_tester {

using parameters = enyx_memory_tester_parameters;

/**
 * The main memory tester object.
 *
 * This wrapper provides @b RAII of @b C @ref enyx_memory_tester
 */
class memory_tester {
public:
    memory_tester(::enyx_memory_tester * handle);

    /**
     * Create the memory tester object from a ``enyx_memory_tester_v2`` HW
     * core.
     *
     * @param core The ``enyx_memory_tester_v2`` core to use
     */
    memory_tester(hw::core & core);

    /**
     * Set memory test parameters.
     *
     * @param params The memory tester parameters to set
     * @return A result object containing an error on failure
     */
    result<void>
    set_parameters(parameters const & params);

    /**
     * Start a memory test run.
     *
     * To configure the test run, use the @ref enyx_memory_tester_set_parameters
     * function with the desired parameters.
     *
     * @return A result object containing an error on failure
     */
    result<void>
    start();

    /**
     * Stop a running memory test.
     *
     * @return A result object containing an error on failure
     */
    result<void>
    stop();

    /**
     * Clear current errors.
     *
     * @return A result object containing an error on failure
     */
    result<void>
    clear_error();

    /**
     * Check if a memory test have finished.
     *
     * @return A result object containing the memory test status on success or
     * an error on failure.
     */
    result<bool>
    is_done() const;

    /**
     * Check if a memory test has failed.
     *
     * @return A result object containing the memory test failure status on
     * success or an error on failure
     */
    result<bool>
    has_failed() const;

    /**
     * Get the last failed test number.
     *
     * This value is only valid when @ref enyx_memory_tester_has_failed is @b
     * true.
     *
     * @return A result object containing the last failed test number on
     * success or an error on failure
     */
    result<std::uint8_t>
    get_last_fail() const;

    /**
     * Get the number of failed memory tests.
     *
     * @return A result object containing the number of failed test on success
     * or an error on failure
     */
    result<std::size_t>
    get_fail_count() const;

    /**
     * Get a const pointer to the underlying @b C handle
     * @ref enyx_memory_tester.
     *
     * @return A const pointer to a @ref enyx_memory_tester
     */
    ::enyx_memory_tester const *
    handle() const;

    /**
     * Get a pointer to the underlying @b C handle @ref enyx_memory_tester.
     *
     * @return A pointer to a @ref enyx_memory_tester
     */
    ::enyx_memory_tester *
    handle();

private:
    std::unique_ptr<::enyx_memory_tester> memtest_;
};

/**
 * Enumerate and create memory tester objects for all ``enyx_memory_tester_v2``
 * HW cores found.
 *
 * @param root The root core
 * @return A result containing a vector of memory tester objects on success or
 * an error on failure.
 */
result<std::vector<memory_tester>>
enumerate(hw::core const & root);

}

ENYX_CORES_NAMESPACE_END

#include "memory_tester.ipp"
